--[[ 

	DCS - Crew Warnings Script V1.04
	by Draken35
	
	
	This script was developed to provide the capability to the AI crew in single player to alert about missile launches and 
	the aircraft being shot at. It was originally develped for the UH-1H Huey but it can be used in any other aircraft.
	
	
	sound files are named as follows:
		Missile launch warnnings: "m" + number (1 to 12) of clock position + ".ogg" . I.E.: "m3.ogg" = Missile lauch, 3 o'clock
		Taking fire warnnings: "t" + number (1 to 12) of clock position + ".ogg" . I.E.: "t6.ogg" = Taking fire, 6 o'clock
   
	Requires MIST 4.4.90  <https://github.com/mrSkortch/MissionScriptingTools/tree/master>
   
   Special thanks for CFRAG for the mentoring, code and ideas!
   
    Version history
	1.0		09/08/2021	Initial release.
	
	1.01    09/08/2021  - Added unit type, white list and blacklist filters
	
	1.02    09/10/2021  - cws.getClockDirection rewriten by cfrag and some code reorg in prep for better things         
						- fixed bug in blindspot detection 
						- Max spotting distance for missile launch 6000m (configurable)
						- general code optimization/simplification
						
	1.03	09/11/2011	 >>> 20 years, Lest we forget <<<
						- allow to switch slots
						- default probabilities of detection change to 100% (to facilitate testing)
						- default read blind spot change to 0 degrees (to facilitate testing)
						
	1.04	09/12/2011 	- Added missile tracking
						- detection profiles
						- Taking Fire alert per shooter suppresion timer
						-  use mist.message.add for alerts
						---- New sound files.
	
		
--]]

trigger.action.outText( 'Crew Warnings Script - V1.04', 10 )

--############################################################################################################################################
cws  = {}

-- Configuration section

-- Unit filtering (list provided as an example...)
cws.ValidUnitTypes = {
		'UH-1H'
	,	'SA342L'
	,	'SA342M'
	,	'SA342Minigun'
	,	'SA342Mistral'
	,	'C-101CC'
	,	'A-29B'
	,	'PUCARA'
} 														-- Script will work only for these types of units

cws.WhiteListedUnitNames = {''}							-- units allowed to use the script, regardless of unit type
cws.BlackListedUnitNames = {''}							-- units NOT allowed to use the script, regardless of unit type

														
-- Alert configuration
cws.BeepBeforeText = false								-- Enable/Disable beep sound before text. Uses "beep" entry in cws.SoundDictionary 
cws.VoiceAlerts = true 									-- Enable/Disable voice alerts
cws.TextAlerts = true 									-- Enable/Disable text alerts
cws.AlertSuppresionTime = 0								-- in seconds, for AAA fire, per shooter, per player

-- Sound files
cws.SoundDictionary = {}
cws.SoundDictionary["beep"] 			= {soundfile = '204521__redoper__roger-beep.ogg', duration = 0.05 }
cws.SoundDictionary["Taking Fire"] 		= {soundfile = 'd35_takingfire.ogg'				, duration = 0.75 }
cws.SoundDictionary["Missile"] 			= {soundfile = 'd35_missile.ogg'				, duration = 0.50 }
cws.SoundDictionary["Missile Launch"] 	= {soundfile = 'd35_missilelaunch.ogg'			, duration = 0.75 }
cws.SoundDictionary["oclock"] 			= {soundfile = 'd35_oclock.ogg'					, duration = 0.50 }
cws.SoundDictionary["0"] 				= {soundfile = 'd35_0.ogg'						, duration = 0.25 }
cws.SoundDictionary["1"] 				= {soundfile = 'd35_1.ogg'						, duration = 0.25 }
cws.SoundDictionary["2"] 				= {soundfile = 'd35_2.ogg'						, duration = 0.25 }
cws.SoundDictionary["3"] 				= {soundfile = 'd35_3.ogg'						, duration = 0.25 }
cws.SoundDictionary["4"] 				= {soundfile = 'd35_4.ogg'						, duration = 0.25 }
cws.SoundDictionary["5"] 				= {soundfile = 'd35_5.ogg'						, duration = 0.25 }
cws.SoundDictionary["6"] 				= {soundfile = 'd35_6.ogg'						, duration = 0.25 }
cws.SoundDictionary["7"] 				= {soundfile = 'd35_7.ogg'						, duration = 0.25 }
cws.SoundDictionary["8"] 				= {soundfile = 'd35_8.ogg'						, duration = 0.25 }
cws.SoundDictionary["9"] 				= {soundfile = 'd35_9.ogg'						, duration = 0.25 }
cws.SoundDictionary["10"] 				= {soundfile = 'd35_10.ogg'						, duration = 0.25 }
cws.SoundDictionary["11"] 				= {soundfile = 'd35_11.ogg'						, duration = 0.25 }
cws.SoundDictionary["12"] 				= {soundfile = 'd35_12.ogg'						, duration = 0.25 }								-- Enable/Disable text alerts




-- Detection Parameters
cws.MaxMissileDetectionDistance = 6000					-- in meters, max missile launch detection range		
cws.MissileTrackingIntervals = 0.5						-- in seconds, time between tracking checks.
cws.DetectionParameters = {}
-- use "0" in a particular direction to indicate a blind spot
--                                   							1	2	3	4	5	6	7	8	9	10	11	12
cws.DetectionParameters["Default"]				= {	Missile	= {100,100,100,100,100,100,100,100,100,100,100,100} , -- used if no specific unit parameters is found
													AAA		= {100,100,100,100,100,100,100,100,100,100,100,100} }
cws.DetectionParameters["UH-1H"]				= {	Missile	= { 95, 95, 95, 80, 60, 30, 60, 80, 95, 95, 95, 95} , -- used if no specific unit parameters is found
													AAA		= { 95, 95, 95, 80, 60, 30, 60, 80, 95, 95, 95, 95} }													
	
	
--############################################################################################################################################
-- int
cws.eventHandler = {}
cws.AlertSuppresedMatrix = {}
-- Handles all world events
function cws.eventHandler:onEvent(_eventDCS)

    if _eventDCS == nil or _eventDCS.initiator == nil then
        return true
    end

    local status, err = pcall(function(_event)

	-- ### Missile launch
	if _event.id == world.event.S_EVENT_SHOT then
		cws.MissileLaunchEvent(_event)
		
	-- ### Taking fire 
	elseif _event.id == 23  and _event.target ~= nil  then
		cws.TakingFireEvent(_event)
		
	end --  if _event.id == world.event.S_EVENT_SHOT then
  -- ### End event processing

        return true
    end, _eventDCS)

    if (not status) then
        env.error(string.format("CWS:Error while handling event %s", err),false)
    end
end
--############################################################################################################################################
function cws.MissileLaunchEvent(_event)
	local _weapon = _event.weapon
	local _target = _weapon:getTarget() 
	
	if _target ~= nil then -- is a guided weapon
	
		if _target:getPlayerName() then
			local _direction = cws.isMissileDetected(_target, _weapon)
			if  _direction >= 0 then		
				local _text = string.format("Missile Launch %i o'clock",_direction)
				cws.displayMessageToGroup(_target,_text,{"Missile Launch",string.format("%i",_direction),"oclock"})
			else 	-- launch was not detected. Track missile over time
				local params = {}
				params.target = _target
				params.missile = _weapon
				timer.scheduleFunction(cws.TrackMissile, params,  timer.getTime() + cws.MissileTrackingIntervals ) 
			end -- if _direction >= 0 then	
		end
	end --if _target ~= nil then
	return true
end
--############################################################################################################################################
function cws.isMissileDetected(_target, _missile)
	-- get launch direction		
	local _Distance = cws.getDistance(_target:getPoint(),_missile:getPoint() )	
	local _playerHeading =  mist.utils.toDegree(mist.getHeading(_target,true)) -- in degrees
	local _direction = cws.getRelativeDirection(_playerHeading,_target:getPoint(),_missile:getPoint() )
	local _DetectionTable = cws.getMissileDetectionTable(_target:getTypeName())
	local _warning = false
	
	-- chech if launch is within detection range
	if _Distance <= cws.MaxMissileDetectionDistance	then
		_direction = cws.getClockDirection(_direction)
		-- check probablity of detection
		_warning = (math.random (1, 100) <= _DetectionTable[_direction] )					
	end --if _launchDistance <=cws.MaxMissileDetectionDistance	then
	if _warning then
		return _direction
	else	
		return -1
	end
end
--############################################################################################################################################
function cws.TrackMissile(params,time)
	_target = params.target
	_missile = params.missile
	
	if _missile:getPoint() then
		-- missile still in fly, check detection
		local _direction = cws.isMissileDetected(_target, _missile)
		if _direction >= 0 then
			-- calculate clock position	 and anounce detection							
			local _text = string.format("Missile! %i o'clock",_direction)
			cws.displayMessageToGroup(_target,_text,{"Missile",string.format("%i",_direction),"oclock"})
			return nil
		else
			return timer.getTime() + 1 -- try to detect again
		end	
	else
		-- missile not longer exists
		return nil
	end
end
--############################################################################################################################################
function cws.TakingFireEvent(_event)
	local _target =_event.target
	if _target:getPlayerName() then
		-- taking fire
		local _playerHeading =  mist.utils.toDegree(mist.getHeading(_target,true)) -- in degrees
		local _direction = cws.getRelativeDirection(_playerHeading,_target:getPoint(),_event.initiator:getPoint() )
		local _DetectionTable = cws.getAAADetectionTable(_target:getTypeName())
		local _warning = false
			
		_direction = cws.getClockDirection(_direction)
		-- check probablity of detection
		_warning = (math.random (1, 100) <= _DetectionTable[_direction] )
		-- announce detection	
		if _warning and not isShooterSuppresed(_target:getName(),_event.initiator:getName()) then 
			local _text = string.format("Taking fire! %i o'clock",_direction)
			cws.displayMessageToGroup(_target,_text,{"Taking Fire",string.format("%i",_direction),"oclock"})
			-- add to the alert supression list
			addSupressedShooter(_target:getName(),_event.initiator:getName()) 
		end 
	end	--if target:getPlayerName() then
	return true
end
--############################################################################################################################################
function cws.getDistance(_point1, _point2)

    local xUnit = _point1.x
    local yUnit = _point1.z
    local xZone = _point2.x
    local yZone = _point2.z

    local xDiff = xUnit - xZone
    local yDiff = yUnit - yZone

    return math.sqrt(xDiff * xDiff + yDiff * yDiff)
end
--############################################################################################################################################
function cws.getRelativeDirection(_refHeading,_point1, _point2)
    local tgtBearing
    local xUnit = _point1.x
    local yUnit = _point1.z
    local xZone = _point2.x
    local yZone = _point2.z
    
    local xDiff = xUnit - xZone
    local yDiff = yUnit - yZone
    
    local tgtAngle = math.deg(math.atan(yDiff/xDiff))
    
    if xDiff > 0 then 
    tgtBearing = 180 + tgtAngle 
    end
    
    if xDiff < 0 and tgtAngle > 0 then
		tgtBearing = tgtAngle 
    end
    
    if xDiff < 0 and tgtAngle < 0 then
		tgtBearing = 360 + tgtAngle
    end   
	
	tgtBearing = tgtBearing - _refHeading
	if tgtBearing > 360 then
		tgtBearing = tgtBearing - 360
	end
	if tgtBearing < 0 then
		tgtBearing =  360 + tgtBearing 
	end
	
    return tgtBearing
end
--############################################################################################################################################
function cws.getClockDirection(_direction)
	-- by cfrag
	if not _direction then return 0 end
	while _direction < 0 do 
		_direction = _direction + 360
	end
		
	if _direction < 15 then -- special case 12 o'clock past 12 o'clock
		return 12
	end
	
	_direction = _direction + 15 -- add offset so we get all other times correct
	return math.floor(_direction/30)
	
end
--############################################################################################################################################
function cws.displayMessageToGroup(_unit, _text, _soundTable)
	if cws.isValidUnitType(_unit:getName(), _unit:getTypeName() ) then
		local msg = {} 
		if cws.TextAlerts then
			msg.text = _text
		else
			msg.text = ""
		end
		msg.displayTime = 10  
		msg.msgFor = {units = {_unit:getName()}} 
		-- compose sound
		msg.multSound = {}
		if cws.VoiceAlerts  then
			local _time = 0
			local _soundindex = 1
			if cws.BeepBeforeText  then
				msg.multSound [_soundindex] = {time = _time, file = cws.SoundDictionary["beep"].soundfile}
				_time = _time +cws.SoundDictionary["beep"].duration
				_soundindex = _soundindex + 1
			end
			
			for _,_sound in pairs(_soundTable) do
				 msg.multSound [_soundindex] = {time = _time, file = cws.SoundDictionary[_sound].soundfile}
				_time = _time +cws.SoundDictionary[_sound].duration
				_soundindex = _soundindex + 1
			end	
		end -- if cws.VoiceAlerts  then
		
		mist.message.add(msg)	
	end -- if cws.isValidUnitType(_unit:getName(), _unit:getTypeName() ) then
end
--############################################################################################################################################
function cws.isValidUnitType (unitName, unitType)
	local found = false
	local validUnit = false
	-- if in BlackListedUnitNames then return false
	for _,v in pairs(cws.BlackListedUnitNames) do
		if v == unitName then
			 found = true
			 validUnit = false
			break
		end
	end
	if not found then -- in the black list		
		for _,v in pairs(cws.WhiteListedUnitNames) do
			if v == unitName then
				 found = true
				 validUnit = true
				break
			end
		end
		if not found then -- in the white list		
			for _,v in pairs(cws.ValidUnitTypes) do
				if v == unitType then
					 validUnit = true
					 found = true
					break
				end
			end
		end -- if not found then -- in the white list	
	end -- if not found then -- in the black list	
	return validUnit
end
--############################################################################################################################################
function cws.getMissileDetectionTable(unitType)
	local DetTable = cws.DetectionParameters[unitType]
	if DetTable == nil then
		DetTable = cws.DetectionParameters['Default']	
	end
	return DetTable.Missile
end
--############################################################################################################################################
function cws.getAAADetectionTable(unitType)
	local DetTable = cws.DetectionParameters[unitType]
	if DetTable == nil then
		DetTable = cws.DetectionParameters['Default']	
	end
	return DetTable.AAA
end
--############################################################################################################################################
function addSupressedShooter(_targetName, _shooterName)
	local valueRow = cws.AlertSuppresedMatrix[_targetName]
	if valueRow == nil then
		cws.AlertSuppresedMatrix[_targetName] = {}
	end
	cws.AlertSuppresedMatrix[_targetName][_shooterName] = timer.getTime()
	return true
end
--############################################################################################################################################
function isShooterSuppresed(_targetName, _shooterName)

	local valueRow = cws.AlertSuppresedMatrix[_targetName]
	local value = nil
	
	if valueRow ~= nil then
		value = valueRow[_shooterName]
	end

	if  value == nil then
		return false
	else
		local t = timer.getTime()
		if t >= (value + cws.AlertSuppresionTime) then
			return false
		else
			return true
		end
	end
end
--############################################################################################################################################




world.addEventHandler(cws.eventHandler)

