#include "common/states11.hlsl"
#include "common/samplers11.hlsl"
#include "common/context.hlsl"
#include "common/random.hlsl"
#include "deferred/GBuffer.hlsl"
#include "deferred/atmosphere.hlsl"
#include "deferred/shading.hlsl"

#define USE_GS 1

// I think this might be the resolution of the screen taking into account things like supersampling or weird render scaling.
uint2 targetDims;

struct VS_INPUT {
	float3 pos		: POSITION0;
	float4 color	: COLOR0;
};

struct VS_OUTPUT {
	float4 sv_pos		: SV_POSITION0;
	float3 wpos			: TEXTURE0;
	float4 projPos	: TEXTURE1;
	float4 color		: COLOR0;
};

VS_OUTPUT VS(VS_INPUT i) {
	VS_OUTPUT o;
	o.sv_pos = o.projPos = mul(float4(i.pos,1), gViewProj); // View space position
	o.wpos = i.pos; // World position
	o.color = i.color;
	return o;
}

float inverse_lerp(float min, float max, float val) {
	return (val - min) / (max - min);
}

float remap(float fromMin, float fromMax, float toMin, float toMax, float value) {
	float relative = inverse_lerp(fromMin, fromMax, value);
	return lerp(toMin, toMax, relative);
}

float4 PS(VS_OUTPUT i): SV_TARGET0 {
//	return float4(applyAtmosphereLinear(gCameraPos.xyz, i.wpos, i.projPos, i.color.rgb*gSunDiffuse.rgb), i.color.a);

	float3 v = normalize(i.wpos - gCameraPos);
#ifdef VERSION_NEWER_2_5_6
	float2 cloudsShadowAO = SampleShadowClouds(i.wpos);
	float3 c = ShadeSolid(i.wpos, gSunDiffuse, i.color.rgb, 0, float3(0,-1,0), 1, 0, cloudsShadowAO.x, 1, cloudsShadowAO, v, 1, FAR_ENV_MAP);
#else
	float3 c = ShadeSolid(gSunDiffuse, i.color.rgb, 0, float3(0, -1, 0), 1, 0, 1, 1, v, 1, FAR_ENV_MAP);
#endif
	return float4(applyAtmosphereLinear(gCameraPos.xyz, i.wpos, i.projPos, c), i.color.a);
}

[maxvertexcount(6)]
void GS(point VS_OUTPUT points[1], inout TriangleStream<VS_OUTPUT> output) {
	float4 p = points[0].sv_pos;
	p /= p.w;

	// Scale the opacity of the dot exponentially with a max visible range of ~22 miles
	// and full opaque at ~5 miles. Since this scales exponentially, by ~10 miles the
	// opacity has already dropped to 50%, making it very difficult to see.
	float3 vecFromCamera = points[0].wpos - gCameraPos;
	float distance = length(vecFromCamera);
	float distanceFactor = saturate(1.0 - inverse_lerp(8000, 42000, distance));
	points[0].color.rgb = points[0].color.rgb;
	points[0].color.a = pow(distanceFactor, 2);

	// Scale dot size with resolution. A baseline of 1 pixel at 1920x1080 is used, and scales
	// up linearly, snapping to whole numbers since you cannot have a "half-pixel." Resolutions
	// below 1080 cannot be scaled down, and so will still see potentially larger dots, but
	// resolutions above 1080 should be roughly normalized to how big a 1080 dot will be, a
	// size that feels easy to enough to track WVR, but not so small that it's easily lost.
	// For common resolutions the pixel math works out to:
	// 1080: 1 pixel
	// 1440: 2 pixels
	// 2160: 3 pixels
	uint controlPixel = 1;
	float2 onePixelSize = float2(1 / gScreenWidth, 1 / gScreenHeight);
	float sizeMultiplier = ceil((gScreenHeight * 0.9) / 1080);

	// The entire screen in clip space actually goes from -1 to 1, a distance of 2.
	float2 scaledDotSize = onePixelSize * sizeMultiplier * controlPixel * 2;

	// Don't allow for anything less than one pixel in size, or else you get nasty flickering
	// from it trying to render the model underneath.
	float2 dotSize = max(onePixelSize, scaledDotSize);

	VS_OUTPUT v[4];
	v[0] = v[1] = v[2] = v[3] = points[0];

	// The p.xy is the precise position of the object in clip space. Since the geometry is built
	// such that this is the top right corner of the object, it needs to be offset by half the
	// dot size. Do some goofy maths to snap the pixels to the pixel grid as best I know how.
	float4 dotPosition;
	float2 pixelPosition = float2(
		floor(remap(-1, 1, 0, gScreenWidth, p.x + dotSize.x / 2.0)),
		floor(remap(-1, 1, 0, gScreenHeight, p.y + dotSize.y / 2.0))
	);
	dotPosition.xy = float2(
		remap(0, gScreenWidth, -1.0, 1.0, pixelPosition.x),
		remap(0, gScreenHeight, -1.0, 1.0, pixelPosition.y)
	);
	dotPosition.z = p.z;
	dotPosition.w = 1;

	v[0].sv_pos = v[1].sv_pos = v[2].sv_pos = v[3].sv_pos = dotPosition;
	v[1].sv_pos.x -= dotSize.x;
	v[2].sv_pos.y -= dotSize.y;
	v[3].sv_pos.x -= dotSize.x;
	v[3].sv_pos.y -= dotSize.y;

	output.Append(v[0]);
	output.Append(v[1]);
	output.Append(v[2]);
	output.RestartStrip();

	output.Append(v[2]);
	output.Append(v[1]);
	output.Append(v[3]);
	output.RestartStrip();
}

technique10 dots
{
    pass P0
	{
		SetVertexShader(CompileShader(vs_5_0, VS()));
		SetPixelShader(CompileShader(ps_5_0, PS()));
#if USE_GS
		SetGeometryShader(CompileShader(gs_5_0, GS()));
#else
		SetGeometryShader(NULL);
#endif
		SetHullShader(NULL);
		SetDomainShader(NULL);
		SetComputeShader(NULL);
		SetDepthStencilState(enableDepthBuffer, 0);
		SetBlendState(enableAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetRasterizerState(cullNone);
    }
}
