local base = _G

module('me_predicates', package.seeall)

local Button = require('Button')
local Container = require('Container')
local ComboBox = require('ComboBox')
local CheckBox = require('CheckBox')
local EditBox = require('EditBox')
local MEditBox = require('MEditBox')
local ListBox = require('ListBox')
local SpinBox = require('SpinBox')
local Widget = require('Widget')
local loader = require('dialog_loader')
local U = require('me_utilities')
local Mission = require('me_mission')
local OpenFileDlg = require('me_file_dialog')
local i18n = require('i18n')
local DB = require('me_db_api');

i18n.setup(_M)

local SPINBOX_MAX_VALUE = 1000000000 --    ,       


function coalitionLister()
    return {{id="red",name=_("RED"),},{id="blue",name=_("BLUE"),},{id="offline",name=_("OFFLINE"),},}
end

function coalitionLister2()
    return {{id=0,name=_("NEUTRAL"),},{id=1,name=_("RED"),},{id=2,name=_("BLUE"),},}
end

function airdromeLister()
	local result = {}
    for __, v in pairs(DB.db.Airports.Airport) do
		table.insert(result, {id=v.WorldID,name=v.Name})
    end
	return result
end

function helipadLister()
	local result = {}
    for __, coalition in pairs(Mission.mission.coalition) do
        for __, country in  ipairs(coalition.country) do
		    for __, group in ipairs(country.static.group) do 
				for __, unit in ipairs(group.units) do
					if unit.type == _("FARP") then
						table.insert(result, {id=unit.unitId,name=unit.name})
					end
				end;
			end;        
		end;
	end;
	return result
end;

function coalitionIdToName(cdata, id)
    if id == "" then return "" elseif id == "red" then return _("RED") elseif id == "blue" then return _("BLUE") elseif id == "offline" then return _("OFFLINE") else return _("UNKNOWN") end
end

function coalitionIdToName2(cdata, id)
    if id == "" then return "" elseif id == 0 then return _("NEUTRAL") elseif id == 1 then return _("RED") elseif id == 2 then return _("BLUE") else return _("UNKNOWN") end
end

function airdromeIdToName(cdata, id)
    for _tmp, v in pairs(DB.db.Airports.Airport) do
		if v.WorldID == id then return v.Name end;
    end;
	return "INVALID AIRDROME ID";
end

function helipadIdToName(cdata, id)
    for __, coalition in pairs(Mission.mission.coalition) do
        for __, country in  ipairs(coalition.country) do
		    for __, group in ipairs(country.static.group) do 
				for __, unit in ipairs(group.units) do
					if unit.unitId == id then return unit.name end
				end;
			end;        
		end;
	end;

	return "INVALID HELIPAD ID";
end

-- groups enumerator
function groupsLister()
    local groups = { }
    for k, v in pairs(Mission.group_by_id) do
        if v then
            table.insert(groups, { id=k, name=v.name })
        end
    end
    table.sort(groups, U.namedTableComparator)
    return groups
end

-- units enumerator
function unitsLister()
    local units = { }
    for k, v in pairs(Mission.unit_by_id) do
        if v then
            table.insert(units, { id=k, name=v.name })
        end
    end
    table.sort(units, U.namedTableComparator)
    return units
end

-- conver unitId to unit name
function unitIdToName(cdata, id)
    return id and Mission.unit_by_id[id] and Mission.unit_by_id[id].name or ""
end

-- conver groupId to unit name
function groupIdToName(cdata, id)
    return id and Mission.group_by_id[id] and Mission.group_by_id[id].name or ""
end


-- return true if unit with specified ID exists
function isUnitExists(id)
    return nil ~= Mission.unit_by_id[id]
end

function isGroupExists(id)
    return nil ~= Mission.group_by_id[id]
end


-- zones enumerator
function zonesLister()
    local zones = { }
    for k, v in pairs(Mission.zone_by_id) do
        if v then
            table.insert(zones, { id=k, name=v.name })
        end
    end
    table.sort(zones, U.namedTableComparator)
    return zones
end



-- conver zoneId to zone name
function zoneIdToName(cdata, id)
    if not id then
        return ""
    else
        local zone = Mission.zone_by_id[id]
        if zone then
            return zone.name
        else
            return ""
        end
    end
end

-- return true if zone with specified ID exists
function isZoneExists(id)
    return nil ~= Mission.zone_by_id[id]
end

function null_transform(cdata, param) -- to make sure we will use serializeFunc instead of displayFunc
    return param 
end 

-- list of available rules for score calculation their arguments
rulesDescr = {
    {
        name = "c_unit_damaged",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            }
        }
    },
    {
        name = "c_unit_alive",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            }
        }
    },
    {
        name = "c_unit_dead",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            }
        }
    },
    {
        name = "c_group_alive",
        fields = {
            {
                id = "group",
                type = "combo",
                comboFunc = groupsLister,
                displayFunc = groupIdToName,
                existsFunc = isGroupExists,
                serializeFunc = groupIdToName,
                default = "",
            }
        }
    },
    {
        name = "c_group_dead",
        fields = {
            {
                id = "group",
                type = "combo",
                comboFunc = groupsLister,
                displayFunc = groupIdToName,
                existsFunc = isGroupExists,
                serializeFunc = groupIdToName,
                default = "",
            }
        }
    },
    {
        name = "c_time_after",
        fields = {
            {
                id = "seconds",
                type = "spin",
                default = 10,
                min = 1,
                max = SPINBOX_MAX_VALUE,
                step = 1
            }
        }
    },
    {
        name = "c_time_before",
        fields = {
            {
                id = "seconds",
                type = "spin",
                default = 10,
                min = 1,
                max = SPINBOX_MAX_VALUE,
                step = 1
            }
        }
    },
    {
        name = "c_flag_is_true",
        fields = {
            {
                id = "flag",
                type = "spin",
                default = 1,
                min = 1,
                max = 1000000000,
            }
        }
    },
    {
        name = "c_flag_is_false",
        fields = {
            {
                id = "flag",
                type = "spin",
                default = 1,
                min = 1,
                max = 1000000000,
            }
        }
    },
    {
        name = "c_time_since_flag",
        fields = {
            {
                id = "flag",
                type = "spin",
                default = 1,
                min = 1,
                max = 1000000000,
            },
            {
                id = "seconds",
                type = "spin",
                default = 10,
                min = 1,
                max = SPINBOX_MAX_VALUE,
                step = 1
            }
        }
    },
    {
        name = "c_unit_in_zone",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "zone",
                type = "combo",
                comboFunc = zonesLister,
                displayFunc = zoneIdToName,
                existsFunc = isZoneExists,
                default = "",
            }
        }
    },
    {
        name = "c_unit_out_zone",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "zone",
                type = "combo",
                comboFunc = zonesLister,
                displayFunc = zoneIdToName,
                existsFunc = isZoneExists,
                default = "",
            }
        }
    },
    {
        name = "c_unit_in_zone_unit",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "zone",
                type = "combo",
                comboFunc = zonesLister,
                displayFunc = zoneIdToName,
                existsFunc = isZoneExists,
                default = "",
            },
            {
                id = "zoneunit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
        }
    },
    {
        name = "c_unit_out_zone_unit",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "zone",
                type = "combo",
                comboFunc = zonesLister,
                displayFunc = zoneIdToName,
                existsFunc = isZoneExists,
                default = "",
            },
            {
                id = "zoneunit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
        }
    },
    {
        name = "c_random_less",
        fields = {
            {
                id = "percent",
                type = "spin",
                default = 10,
            }
        }
    },
    {
        name = "c_unit_altitude_higher",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "altitude",
                type = "positive_number",
                default = 1,
            },
        }
    },
    {
        name = "c_unit_altitude_lower",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "altitude",
                type = "positive_number",
                default = 1,
            },
        }
    },
    {
        name = "c_unit_speed_higher",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "speed",
                type = "positive_number",
                default = 100,
            },
        }
    },
    {
        name = "c_unit_speed_lower",
        fields = {
            {
                id = "unit",
                type = "combo",
                comboFunc = unitsLister,
                displayFunc = unitIdToName,
                existsFunc = isUnitExists,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "speed",
                type = "positive_number",
                default = 100,
            },
        }
    },
    {
        name = "c_mission_score_higher",
        fields = {
            {
                id = "coalitionlist",
                type = "combo",
                comboFunc = coalitionLister,
                displayFunc = coalitionIdToName,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "score",
                type = "spin",
                default = 50,
                min = -100,
                max = 100,
            },
        }
    },
    {
        name = "c_coalition_has_airdrome",
        fields = {
            {
                id = "coalitionlist",
                type = "combo",
                comboFunc = coalitionLister2,
                displayFunc = coalitionIdToName2,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "airdromelist",
                type = "combo",
                comboFunc = airdromeLister,
                displayFunc = airdromeIdToName,
                serializeFunc = null_transform,
                default = "",
            },
        }
    },
    {
        name = "c_coalition_has_helipad",
        fields = {
            {
                id = "coalitionlist",
                type = "combo",
                comboFunc = coalitionLister2,
                displayFunc = coalitionIdToName2,
                serializeFunc = null_transform,
                default = "",
            },
            {
                id = "helipadlist",
                type = "combo",
                comboFunc = helipadLister,
                displayFunc = helipadIdToName,
                serializeFunc = null_transform,
                default = "",
            },
        }
    },
};


cdata = {

    predicates = {
        c_unit_alive = _("UNIT ALIVE"),
        c_unit_dead = _("UNIT DEAD"),
        c_group_alive = _("GROUP ALIVE"),
        c_group_dead = _("GROUP DEAD"),
        c_unit_damaged = _("UNIT DAMAGED"),
        c_time_after = _("TIME MORE"),
        c_time_before = _("TIME LESS"),
        c_unit_in_zone = _("UNIT IN ZONE"),
        c_unit_out_zone = _("UNIT OUTSIDE ZONE"),
        c_unit_in_zone_unit = _("UNIT IN MOVING ZONE"),
        c_unit_out_zone_unit = _("UNIT OUTSIDE OF MOVING ZONE"),
        c_flag_is_true = _("FLAG IS TRUE"),
        c_flag_is_false = _("FLAG IS FALSE"),
        c_time_since_flag = _("TIME SINCE FLAG"),
        c_random_less = _("RANDOM"),
        c_unit_altitude_higher = _("UNIT'S ALTITUDE HIGHER THAN"),
        c_unit_altitude_lower = _("UNIT'S ALTITUDE LOWER THAN"),
        c_unit_speed_higher = _("UNIT'S SPEED HIGHER THAN"),
        c_unit_speed_lower = _("UNIT'S SPEED LOWER THAN"),
        c_mission_score_higher = _("MISSION SCORE HIGHER THAN"),
		c_coalition_has_airdrome = _("COALITION HAS AIRDROME"),
		c_coalition_has_helipad = _("COALITION HAS HELIPAD"),
    },

    values = {
        unit = _("UNIT:"),
        group = _("GROUP:"),
        zone = _("ZONE:"),
        seconds = _("SECONDS:"),
        flag = _("FLAG:"),
        percent = "%",
        zoneunit = _("ZONE UNIT:"),
        coalitionlist = _("COALITION:"),
        altitude = _("ALTITUDE:"),
        speed = _("SPEED:"),
        airdromelist = _("AIRDROME:"),
		helipadlist = _("FARP:"),
    }

}


-- return index of value in table
function getIndex(table, value)
  local idx = 1 
  for _tmp, v in ipairs(table) do
      if v == value then
          return idx
      end
      idx = idx + 1
  end
  return idx
end


-- add rules to the list
function rulesToList(list, rules, cdata)
  list:clear()
  if rules then
      for _tmp, v in ipairs(rules) do
          U.addListBoxItem(list, getRuleAsText(v, cdata), nil, v)
      end
  end
  list:onChange(nil)
end

function rulesToList2(listA, listB, rules, cdata)
  listA:clear()
  listB:clear()
  if rules then
      for _tmp, v in ipairs(rules) do
          if v.logicBlock == "B" then
			U.addListBoxItem(listB, getRuleAsText(v, cdata), nil, v)
		  else
			U.addListBoxItem(listA, getRuleAsText(v, cdata), nil, v)
		  end
      end
  end
  listA:onChange(nil)
end

-- build text from rule
function getRuleAsText(rule, cdata)
    local ruleDescr = rule.predicate
    local str = getPredicateName(ruleDescr, cdata) .. ' ('

    local first = true
    local needComma = false
    for _tmp, field in ipairs(ruleDescr.fields) do
        if needComma then
            str = str .. ', '
        else
            needComma = true
        end

        local text
        if field.displayFunc then
            text = field.displayFunc(cdata, rule[field.id])
        else
            text = tostring(rule[field.id])
        end

        if first and ruleDescr.firstValueAsName then
            str = text .. ' ('
            needComma = false
        else
            str = str .. text
        end

        first = false
    end

    str = str .. ')'
    return str
end

-- returns localized version of getRuleAsText function
function ruleTextFunc(cdata)
    return function (rule) 
        return getRuleAsText(rule, cdata) 
    end
end



-- update row in list
function updateListRow(list, displayFunc)
    local item = list:getSelectedItem()
    if item then
        local struct = item.itemId
        item:setText(displayFunc(struct))
    end
end


-- gets structure from list, sets value in structure and update row in list
-- in order to get value it uses function getValueFunc(control)
function bindAbstractValue(control, list, valueName, displayFunc, 
                 getValueFunc)
    function control:onChange(itemValue)
        local item = list:getSelectedItem()
        if item then
            local struct = item.itemId
            struct[valueName] = getValueFunc(control, itemValue)
            updateListRow(list, displayFunc)
        end
    end
end

-- links value in text editor control, value in structure and list box
function bindTextValue(control, list, valueName, displayFunc)
    bindAbstractValue(control, list, valueName, displayFunc,
        function (c, itemValue) 
            local text = c:getText();
            return text; 
        end)
end


-- links value in combo box, value in structure and list box
function bindComboValue(control, list, valueName, displayFunc)
    bindAbstractValue(control, list, valueName, displayFunc,
        function (combo, item)
            if not combo.nameValue then
                return combo:getText()
            else
                return item.id
            end
        end)
end


-- links value in spinbox control, value in structure and list box
function bindNumValue(control, list, valueName, displayFunc)
    bindAbstractValue(control, list, valueName, displayFunc,
        function (c) 
            return c:getValue(); 
            end)
end

-- links value in spinbox control, value in structure and list box
function bindNumValuePositive(control, list, valueName, displayFunc)
    bindAbstractValue(control, list, valueName, displayFunc,
        function (c) 
            local text = c:getText();
            local number = tonumber(text);
            if (text ~= nil) and (number ~= nil) and (number > 0) then
                return text;
            else
                return '0';
            end
        end)
end

-- links value in checkbox control, value in structure and list box
function bindCheckboxValue(control, list, valueName, displayFunc)
    bindAbstractValue(control, list, valueName, displayFunc,
        function (c) return c:getState(); end)
end

-- returns display name of predicate
function getPredicateName(desc, cdata)
    return cdata.predicates[desc.name]
end

-- fill predicates combo with predicate values
function fillPredicatesCombo(combo, descr, cdata)
    combo:clear()
    combo:setReadonly(true);
    local first = nil
    for _tmp, v in ipairs(descr) do
        local item = U.addComboBoxItem(combo, getPredicateName(v, cdata), nil, v)
        if not first then
            first = item
        end
    end
    if first then
        combo.selectedItem = first.itemId
        combo:setText(getPredicateName(first.itemId, cdata))
    end
end


-- fill dictionary
function fillDict(combo, fun)
    if fun then
        local tbl = fun()
        for _tmp, v in ipairs(tbl) do
            if type(v) == 'table' then
                combo.nameValue = true
                local item = combo:newItem(v.name)
                item.id = v.id
                item.name = v.name
                combo:insertWidget(item)
            else
                combo.nameValue = false
                U.addComboBoxItem(combo, v)
            end
        end
    end
end


-- set value of combo box
-- if combo box works with tuple, finds text in tuples list
function setComboValue(combo, value)
    if not combo.nameValue then
        combo:setText(value)
    else
        local items = combo:getWidgets()
        for _tmp, v in pairs(items) do
            if v.id == value then
                combo:setText(v:getText())
                return
            end
        end
    end
end

-- create widgets for predicate editing
function updateArgumentsPanel(rule, list, ctr, labelX, comboX, labelW, 
        comboW, comboH, cdata, labelTheme)
    local ruleDescr = rule.predicate

    ctr:removeAllChildren()
    
    local offsetX, _tmp, _tmp, _tmp = ctr:getBounds()
    labelX = labelX - offsetX
    comboX = comboX - offsetX

    local y = 1
    for _tmp, field in ipairs(ruleDescr.fields) do
        local _x, _y, _w, _h = comboX, y, comboW, comboH;
        local label = Widget.new(cdata.values[field.id])
        label:setBounds(labelX, y, labelW, comboH)
        label:setTheme(labelTheme)
        ctr:addChild(label)
    
        local control
        if (field.type == "combo") then
            control = ComboBox.new()
            control:setReadonly(true)
            bindComboValue(control, list, field.id, ruleTextFunc(cdata))
            fillDict(control, field.comboFunc)
            setComboValue(control, rule[field.id])
        elseif (field.type == "spin") then
            control = SpinBox.new()
            if (nil ~= field.min) and (nil ~= field.max) then
                control:setRange(field.min, field.max)
            end
            if nil ~= field.step then
                control:setStep(field.step)
            end
            bindNumValue(control, list, field.id, ruleTextFunc(cdata))
            control:setValue(rule[field.id])
        elseif (field.type == "positive_number") then
            control = EditBox.new()
            control:setNumber(true);
            bindNumValuePositive(control, list, field.id, ruleTextFunc(cdata))
            control:setText(rule[field.id])
        elseif (field.type == "checkbox") then
            control = CheckBox.new()
            bindCheckboxValue(control, list, field.id, ruleTextFunc(cdata))
            control:setState(rule[field.id])
        elseif (field.type == "medit") then
            control = MEditBox.new()
            bindTextValue(control, list, field.id, ruleTextFunc(cdata))
            --bindTextValue(control, list, field.id, field.displayFunc)
            control:setText(rule[field.id])
            _x, _y, _w, _h = comboX, y, comboW, 5*comboH;
        elseif (field.type == "file_edit") then
            local container = Container.new();
            local edit = EditBox.new()
            control = edit;
            bindTextValue(control, list, field.id, ruleTextFunc(cdata))
            control:setText(rule[field.id])
            control:setBounds(0, 0, comboW*0.69, comboH)
            container:addChild(control)
            control = Button.new()
            control.edit = edit;
            bindAbstractValue(control, list, field.id, ruleTextFunc(cdata),
                                chooseFile);

            control:setBounds(comboW*0.71, 0, comboW*0.29, comboH)
            control:setText(cdata.open);
            container:addChild(control)
            control = container;
        elseif (field.type == "file_miz") then
            local container = Container.new();
            local edit = EditBox.new()
            control = edit;
            bindTextValue(control, list, field.id, ruleTextFunc(cdata))
            control:setText(rule[field.id])
            control:setBounds(0, 0, comboW*0.69, comboH)
            container:addChild(control)
            control = Button.new()
            control.edit = edit;
            bindAbstractValue(control, list, field.id, ruleTextFunc(cdata),
                                chooseMiz);

            control:setBounds(comboW*0.71, 0, comboW*0.29, comboH)
            control:setText(cdata.open);
            container:addChild(control)
            control = container;
        else
            control = EditBox.new()
            bindTextValue(control, list, field.id, ruleTextFunc(cdata))
            control:setText(rule[field.id])
        end
        control:setBounds(_x, _y, _w, _h)
        ctr:addChild(control)
  
        y = _y + _h + 2
    end

end


-- copy default values from rule descriptions
function setRuleDefaults(rule, descr)
    for _tmp, v in ipairs(descr.fields) do
        if not rule[v.id] then
            rule[v.id] = v.default
        end
    end
end


-- create new rule from description
function createRule(descr)
    local res = { predicate = descr; }
    setRuleDefaults(res, descr)
    return res
end


-- show or hide widgets (passed as variable length arguments) and remove
-- all children from container
function showWidgets(visible, container, ...)
    for _tmp, v in ipairs(arg) do
        v:setVisible(visible)
    end
    if not visible then
        container:removeAllChildren()
    end
end


-- index rules by name
function getRulesIndex(rules)
    local result = { }
    for _tmp, rule in ipairs(rules) do
        result[rule.name] = rule
    end
    return result
end


-- convert rule to function call
function actionToString(rule)
    --print('rule', rule.predicate.name)
    local str = rule.predicate.name .. '(';
    local firstArg = true
    for _tmp, field in ipairs(rule.predicate.fields) do
        if not firstArg then
          str = str .. ', '
        else
          firstArg = false
        end
        local value = rule[field.id]
        if field.serializeFunc then
            value = field.serializeFunc(cdata, rule[field.id])
        elseif field.displayFunc then
            value = field.displayFunc(cdata, rule[field.id])
        end
        if not value then value = "" end -- fix for empty fields
        str = str .. string.format('%q', base.tostring(value))
    end
    str = str .. ')'
    return str
end

function chooseFile(c)
--    print(c:getText());
    OpenFileDlg.resize( 150, 150, 500, 500);
    OpenFileDlg.setStyle('sound', _('Choose sound file'), OpenFileDlgCallback, c.edit);
    OpenFileDlg.show(true);
end;

function chooseMiz(c)
--    print(c:getText());
    OpenFileDlg.resize( 150, 150, 500, 500);
    OpenFileDlg.setStyle('addmission', _('Choose mission file'), OpenFileDlgCallback_addmiz, c.edit);
    OpenFileDlg.show(true);
end;

function OpenFileDlgCallback(filename, edit)
    if not filename then return; end;
    local str = U.extractFileName(filename)
    --edit:setText(str);
    local oldFileName = edit:getText();
    local res = Mission.insertFileIntoMission(filename, nil, oldFileName);
    if res then 
        edit:setText(str);
        edit:onChange();
    end;
    
end;

function OpenFileDlgCallback_addmiz(filename, edit) -- no inserts, no full path
    if not filename then return; end;
    edit:setText(U.extractFileName(filename));
    edit:onChange();
end;

-- returns true if predicate valid
function isValidRule(rule)
    if not rule or not rule.predicate or not rule.predicate.fields then return false end
    for _tmp, v in pairs(rule.predicate.fields) do
        if v.existsFunc and (not v.existsFunc(rule[v.id])) then
            return false
        end
    end
    return true
end


-- remove all invalid rules from rules list
function removeInvalidRules(rulesList)
    local toRemove = { }

    for i = 1, #rulesList do
        if not isValidRule(rulesList[i]) then
            table.insert(toRemove, i)
        end
    end

    for i = 1, #toRemove do
        table.remove(rulesList, toRemove[i] - (i - 1))
    end
end

