--DiCE: DCS-Integrated-Countermeasure-Editor-hook
function loadDiCE()

local DiCE = {}

local version = 6.0
local versionID = 20230522 -- YYYYMMDD

-- v6 adds the option to make DiCE IC compatible

local Tools = require("tools")
local dcsInstallPath = lfs.currentdir() -- the users dcs install folder
local optionsLuaPath = lfs.writedir() .. 'config\\options.lua'	-- the users options lua file
local optionsLuaPathData -- where the options lua table will be reloaded

-- CMDS file paths for the modules
local cmdsLua_F18C_fullPath = 	dcsInstallPath ..[[Mods\aircraft\FA-18C\Cockpit\Scripts\TEWS\device\CMDS_ALE47.lua]]
local cmdsLua_F16C_fullPath = 	dcsInstallPath ..[[Mods\aircraft\F-16C\Cockpit\Scripts\EWS\CMDS\device\CMDS_ALE47.lua]]
local cmdsLua_A10C_fullPath = 	dcsInstallPath ..[[Mods\aircraft\A-10C\Cockpit\Scripts\AN_ALE40V\device\AN_ALE40V_params.lua]]
local cmdsLua_A10C2_fullPath = 	dcsInstallPath ..[[Mods\aircraft\A-10C_2\Cockpit\Scripts\AN_ALE40V\device\AN_ALE40V_params.lua]]
local cmdsLua_M2000C_fullPath = dcsInstallPath ..[[Mods\aircraft\M-2000C\Cockpit\Scripts\SPIRALE.lua]]
local cmdsLua_AV8B_fullPath = 	dcsInstallPath ..[[Mods\aircraft\AV8BNA\Cockpit\Scripts\EWS\EW_Dispensers_init.lua]]

-- Original CMDS file paths for the modules
local defaultCMDS_F18C = lfs.writedir() .. 'config\\DiCE\\F18C\\CMDS_ALE47.lua'
local defaultCMDS_F16C = lfs.writedir() .. 'config\\DiCE\\F16C\\CMDS_ALE47.lua'
local defaultCMDS_A10C = lfs.writedir() .. 'config\\DiCE\\A10C\\AN_ALE40V_params.lua'
local defaultCMDS_A10C2 = lfs.writedir() .. 'config\\DiCE\\A10C2\\AN_ALE40V_params.lua'
local defaultCMDS_M2000C = lfs.writedir() .. 'config\\DiCE\\M2000C\\SPIRALE.lua'
local defaultCMDS_AV8B = lfs.writedir() .. 'config\\DiCE\\AV8B\\EW_Dispensers_init.lua'


-- Create a log for troubleshooting
local logFile = io.open(lfs.writedir() .. [[Logs\DCS-DiCE.log]], "w")
function log(str)
    if not str then
        return
    end
    if logFile then
        logFile:write("[" .. os.date("%H:%M:%S") .. "] " .. str .. "\r\n")
        logFile:flush()
    end
end
  
-- The following functions are triggers to update the CMDS files
-- Ideally they would update only as options.lua updates
-- Currently they "effectively" update when the player enters the aircraft, or rather
-- that's how it will seem to the player

function DiCE.onShowGameMenu() -- this works best when the ESC menu is OPENED
	log('[DiCE] onShowGameMenu()')
	updateEventDetected()
end

function DiCE.onMissionLoadBegin() -- this is for catching updating files before the mission start
	log('[DiCE] onMissionLoadBegin()')
	updateEventDetected()
end

function DiCE.onSimulationStart() -- this is for catching at the start of the mission
	log('[DiCE] onSimulationStart()')
	updateEventDetected()
end

function DiCE.onShowBriefing() -- this is for catching when the brief is shown. does not always work
	log('[DiCE] onShowBriefing()')
	updateEventDetected()
end


--TODO: Find a function that triggers before the person tries to join a server.
-- Maybe something on the main menu

DCS.setUserCallbacks(DiCE)  -- here we set our callbacks

net.log('[DiCE] v' .. version .. ' Loaded ...')

-- Functions --

function updateEventDetected()
	log('[DiCE] updateEventDetected()')
	-- This is where we can access the options.lua of the player.
	-- after getting the values we will modify the values that can be found
	-- in the players countermeasure files for each aircraft that they have
	if file_exists(optionsLuaPath) == false then
		log('[DiCE] Options.lua not detected. Cannot update countermeasure files.')
		return
	else
		log('[DiCE] Options.lua detected')
	end
	
	optionsLuaPathData = Tools.safeDoFile(optionsLuaPath, false)
	-- Now that the options.lua is detected, read it and determine the values
	-- The format for getting values is, for example
	-- optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramEChaff']
	-- The above would return '2'
	-- optionsLuaPathData['options']['plugins']['DiCE A-10C'] would return the table
	
	-- Determine if the DiCE Disable checkbox is checked
	-- if true, then dice is disabled
	-- if false, CM changes can be made and applied
	local isDiceDisabled = optionsLuaPathData['options']['plugins']['DiCE M2000C']['makeICSafe']
	
	if isDiceDisabled then --if the box is checked, dice is disabled
		log('[DiCE] is disabled. Making aircraft IC compatable')
		if file_exists(cmdsLua_F18C_fullPath) then
			makeIcSafe_F18C()
		end
		if file_exists(cmdsLua_F16C_fullPath) then
			makeIcSafe_F16C()
		end
		if file_exists(cmdsLua_A10C_fullPath) then
			makeIcSafe_A10C()
		end
		if file_exists(cmdsLua_A10C2_fullPath) then
			makeIcSafe_A10C2()
		end
		if file_exists(cmdsLua_M2000C_fullPath) then
			makeIcSafe_M2000C()
		end
		if file_exists(cmdsLua_AV8B_fullPath) then
			makeIcSafe_AV8B()
		end
	else -- dice is not disabled. do the CM stuff
		if file_exists(cmdsLua_F18C_fullPath) then
			writeF18CFile()
		end
		if file_exists(cmdsLua_F16C_fullPath) then
			writeF16CFile()
		end
		if file_exists(cmdsLua_A10C_fullPath) then
			writeA10CFile()
		end
		if file_exists(cmdsLua_A10C2_fullPath) then
			writeA10C2File()
		end	
		if file_exists(cmdsLua_M2000C_fullPath) then
			writeM2000CFile()
		end
		if file_exists(cmdsLua_AV8B_fullPath) then
			writeAV8BFile()
		end
	end --end of ifElse
end -- end of updateEventDetected()


function makeIcSafe_F18C() --https://stackoverflow.com/questions/16367524/copy-csv-file-to-new-file-in-lua
	local infile = io.open(defaultCMDS_F18C, "r")
	local instr = infile:read("*a")
	infile:close()
	
	local outfile = io.open(cmdsLua_F18C_fullPath, "w")
	outfile:write(instr)
	outfile:close()
end

function makeIcSafe_F16C()
	local infile = io.open(defaultCMDS_F16C, "r")
	local instr = infile:read("*a")
	infile:close()
	
	local outfile = io.open(cmdsLua_F16C_fullPath, "w")
	outfile:write(instr)
	outfile:close()
end

function makeIcSafe_A10C()
	local infile = io.open(defaultCMDS_A10C, "r")
	local instr = infile:read("*a")
	infile:close()
	
	outfile = io.open(cmdsLua_A10C_fullPath, "w")
	outfile:write(instr)
	outfile:close()
end

function makeIcSafe_A10C2()
	local infile = io.open(defaultCMDS_A10C2, "r")
	local instr = infile:read("*a")
	infile:close()
	
	local outfile = io.open(cmdsLua_A10C2_fullPath, "w")
	outfile:write(instr)
	outfile:close()
end

function makeIcSafe_M2000C()
	local infile = io.open(defaultCMDS_M2000C, "r")
	local instr = infile:read("*a")
	infile:close()
	
	local outfile = io.open(cmdsLua_M2000C_fullPath, "w")
	outfile:write(instr)
	outfile:close()
	
end

function makeIcSafe_AV8B()
	local infile = io.open(defaultCMDS_AV8B, "r")
	local instr = infile:read("*a")
	infile:close()
	
	local outfile = io.open(cmdsLua_AV8B_fullPath, "w")
	outfile:write(instr)
	outfile:close()
end

--[[ Template for writing to file
function writeTemplateFile()
    log('[DiCE] Update Template CMDS')
	
    local file = nil
    file = io.open(cmdsLua_Template_fullPath, "w")
    
    file:write("local gettext = require(\"i_18n\")\n") -- contents of file, etc...
    
    file:close()
            
    log('[DiCE] Template CMDS Updated')
end
]]


function writeAV8BFile()
    log('[DiCE] Update AV8B CMDS')
	
    local file = nil
    file = io.open(cmdsLua_AV8B_fullPath, "w")
    
    file:write("local gettext = require(\"i_18n\")\n")
    file:write("_ = gettext.translate\n")
    file:write("\n")
    file:write("-- Chaff Burst Values\n")
    file:write("-- BQTY: 1 to 15. Special values: -1 = Continuous (will use ALL chaff); -2 = Random (will dispense between 1 to 6 chaff)\n")
    file:write("-- BINT: 0.1 to 1.5 seconds. Special values: -2 = Random (will set an interval between 0.1 and 0.4 seconds)\n")
    file:write("\n")
    file:write("-- Chaff Salvo Values\n")
    file:write("-- SQTY: 1 to 15.\n")
    file:write("-- SINT: 1 to 15 seconds.\n")
    file:write("\n")
    file:write("-- Flare Salvo Values\n")
    file:write("-- SQTY: 1 to 15.\n")
    file:write("-- SINT: 1 to 15 seconds.\n")
    file:write("\n")
    file:write("--All Expendables\n")
    file:write("EW_ALL_CHAFF_BQTY = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BAllExpendablesChaffBurstQuantity'] .. ";\n")
    file:write("EW_ALL_CHAFF_BINT = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BAllExpendablesChaffBurstInterval'] .. ";\n")
    file:write("EW_ALL_CHAFF_SQTY = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BAllExpendablesChaffSalvoQuantity'] .. ";\n")
    file:write("EW_ALL_CHAFF_SINT = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BAllExpendablesChaffSalvoInterval'] .. ";\n")
    file:write("EW_ALL_FLARES_SQTY = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BAllExpendablesFlaresSalvoQuantity'] .. ";\n")
    file:write("EW_ALL_FLARES_SINT = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BAllExpendablesFlaresSalvoInterval'] .. ";\n")
    file:write("\n")
    file:write("--Chaff Only\n")
    file:write("EW_CHAFF_BQTY = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BChaffOnlyChaffBurstQuantity'] .. ";\n")
    file:write("EW_CHAFF_BINT = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BChaffOnlyChaffBurstInterval'] .. ";\n")
    file:write("EW_CHAFF_SQTY = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BChaffOnlyChaffSalvoQuantity'] .. ";\n")
    file:write("EW_CHAFF_SINT = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BChaffOnlyChaffSalvoInterval'] .. ";\n")
    file:write("\n")
    file:write("--Flares Only\n")
    file:write("EW_FLARES_SQTY = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BFlaresOnlyFlaresSalvoQuantity'] .. ";\n")
    file:write("EW_FLARES_SINT = " .. optionsLuaPathData['options']['plugins']['DiCE AV-8B']['AV8BFlaresOnlyChaffSalvoInterval'] .. ";\n")
    file:write("\n")
    file:write("need_to_be_closed = true\n")
    
    file:close()

    log('[DiCE] AV8B CMDS Updated')
end


function writeM2000CFile()
    log('[DiCE] Update M2000C CMDS')
	
    local file = nil
    file = io.open(cmdsLua_M2000C_fullPath, "w")
    
    file:write("-- You can now add your own RWR customization in Saved Games instead of altering this file:close\n")
    file:write("-- Just copy this file to Saved Games\\DCS(.openbeta)\\Config\\M-2000C\\SPIRALE.lua\n")
    file:write("-- And make your own changes\n")
    file:write("\n")
    file:write("programs = {}\n")
    file:write("programs[ 1] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual01Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual01Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual01Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual01Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual01CycleInterval'] .. "	}\n")
    file:write("programs[ 2] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual02Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual02Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual02Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual02Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual02CycleInterval'] .. "	}\n")
    file:write("programs[ 3] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual03Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual03Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual03Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual03Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual03CycleInterval'] .. "	}\n")
    file:write("programs[ 4] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual04Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual04Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual04Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual04Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual04CycleInterval'] .. "	}\n")
    file:write("programs[ 5] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual05Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual05Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual05Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual05Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual05CycleInterval'] .. "	}\n")
    file:write("programs[ 6] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual06Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual06Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual06Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual06Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual06CycleInterval'] .. "	}\n")
    file:write("programs[ 7] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual07Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual07Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual07Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual07Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual07CycleInterval'] .. "	}\n")
    file:write("programs[ 8] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual08Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual08Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual08Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual08Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual08CycleInterval'] .. "	}\n")
    file:write("programs[ 9] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual09Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual09Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual09Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual09Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual09CycleInterval'] .. "	}\n")
    file:write("programs[10] = { chaff  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual10Chaff'] .. ",	flare  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual10Flare'] .. ",		intv   = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual10Interval'] .. ",	cycle  = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual10Cycle'] .. ",		c_intv = " .. optionsLuaPathData['options']['plugins']['DiCE M2000C']['M2000CManual10CycleInterval'] .. "	}\n")
    
    file:close()

    log('[DiCE] M2000C CMDS Updated')
end


function writeA10C2File()
    log('[DiCE] Update A10C2 CMDS')
	
    local file = nil
    file = io.open(cmdsLua_A10C2_fullPath, "w")
	
    file:write("local gettext = require(\"i_18n\")\n")
    file:write("_ = gettext.translate\n")
    file:write("\n")
    file:write("programs = {}\n")
    file:write("\n")
    file:write("-- Old generation radar SAM\n")
    file:write("programs['A'] = {}\n")
    file:write("programs['A'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramAChaff'] .. "\n")
    file:write("programs['A'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramAFlare'] .. "\n")
    file:write("programs['A'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramAInterval'] .. "\n")
    file:write("programs['A'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramACycle'] .. "\n")
    file:write("\n")
    file:write("-- Current generation radar SAM\n")
    file:write("programs['B'] = {}\n")
    file:write("programs['B'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramBChaff'] .. "\n")
    file:write("programs['B'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramBFlare'] .. "\n")
    file:write("programs['B'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramBInterval'] .. "\n")
    file:write("programs['B'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramBCycle'] .. "\n")
    file:write("\n")
    file:write("-- IR SAM\n")
    file:write("programs['C'] = {}\n")
    file:write("programs['C'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramCChaff'] .. "\n")
    file:write("programs['C'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramCFlare'] .. "\n")
    file:write("programs['C'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramCInterval'] .. "\n")
    file:write("programs['C'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramCCycle'] .. "\n")
    file:write("\n")
    file:write("-- Default manual presets\n")
    file:write("-- Mix 1\n")
    file:write("programs['D'] = {}\n")
    file:write("programs['D'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramDChaff'] .. "\n")
    file:write("programs['D'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramDFlare'] .. "\n")
    file:write("programs['D'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramDInterval'] .. "\n")
    file:write("programs['D'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramDCycle'] .. "\n")
    file:write("\n")
    file:write("-- Mix 2\n")
    file:write("programs['E'] = {}\n")
    file:write("programs['E'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramEChaff'] .. "\n")
    file:write("programs['E'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramEFlare'] .. "\n")
    file:write("programs['E'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramEInterval'] .. "\n")
    file:write("programs['E'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramECycle'] .. "\n")
    file:write("\n")
    file:write("-- Mix 3\n")
    file:write("programs['F'] = {}\n")
    file:write("programs['F'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramFChaff'] .. "\n")
    file:write("programs['F'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramFFlare'] .. "\n")
    file:write("programs['F'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramFInterval'] .. "\n")
    file:write("programs['F'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramFCycle'] .. "\n")
    file:write("\n")
    file:write("-- Mix 4\n")
    file:write("programs['G'] = {}\n")
    file:write("programs['G'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramGChaff'] .. "\n")
    file:write("programs['G'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramGFlare'] .. "\n")
    file:write("programs['G'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramGInterval'] .. "\n")
    file:write("programs['G'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramGCycle'] .. "\n")
    file:write("\n")
    file:write("-- Chaff single\n")
    file:write("programs['H'] = {}\n")
    file:write("programs['H'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramHChaff'] .. "\n")
    file:write("programs['H'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramHFlare'] .. "\n")
    file:write("programs['H'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramHInterval'] .. "\n")
    file:write("programs['H'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramHCycle'] .. "\n")
    file:write("\n")
    file:write("-- Chaff pair\n")
    file:write("programs['I'] = {}\n")
    file:write("programs['I'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramIChaff'] .. "\n")
    file:write("programs['I'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramIFlare'] .. "\n")
    file:write("programs['I'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramIInterval'] .. "\n")
    file:write("programs['I'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramICycle'] .. "\n")
    file:write("\n")
    file:write("-- Flare single\n")
    file:write("programs['J'] = {}\n")
    file:write("programs['J'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramJChaff'] .. "\n")
    file:write("programs['J'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramJFlare'] .. "\n")
    file:write("programs['J'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramJInterval'] .. "\n")
    file:write("programs['J'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramJCycle'] .. "\n")
    file:write("\n")
    file:write("-- Flare pair\n")
    file:write("programs['K'] = {}\n")
    file:write("programs['K'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramKChaff'] .. "\n")
    file:write("programs['K'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramKFlare'] .. "\n")
    file:write("programs['K'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramKInterval'] .. "\n")
    file:write("programs['K'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramKCycle'] .. "\n")
    file:write("\n")
    file:write("-- Chaff pre-empt\n")
    file:write("programs['L'] = {}\n")
    file:write("programs['L'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramLChaff'] .. "\n")
    file:write("programs['L'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramLFlare'] .. "\n")
    file:write("programs['L'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramLInterval'] .. "\n")
    file:write("programs['L'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramLCycle'] .. "\n")
    file:write("\n")
    file:write("-- Flare pre-empt\n")
    file:write("programs['M'] = {}\n")
    file:write("programs['M'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramMChaff'] .. "\n")
    file:write("programs['M'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramMFlare'] .. "\n")
    file:write("programs['M'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramMInterval'] .. "\n")
    file:write("programs['M'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramMCycle'] .. "\n")
    file:write("\n")
    file:write("\n")
    file:write("\n")
    file:write("ContainerChaffCapacity = 120\n")
    file:write("\n")
    file:write("ContainerFlareCapacity = 120\n")
    file:write("\n")
    file:write("NumberOfContiners      = 4\n")
    file:write("\n")
    file:write("AN_ALE_40V_FAILURE_TOTAL = 0\n")
    file:write("AN_ALE_40V_FAILURE_CONTAINER_LEFT_WING	= 1\n")
    file:write("AN_ALE_40V_FAILURE_CONTAINER_LEFT_GEAR	= 2\n")
    file:write("AN_ALE_40V_FAILURE_CONTAINER_RIGHT_GEAR	= 3\n")
    file:write("AN_ALE_40V_FAILURE_CONTAINER_RIGHT_WING	= 4\n")
    file:write("\n")
    file:write("Damage = {	{Failure = AN_ALE_40V_FAILURE_TOTAL, Failure_name = \"AN_ALE_40V_FAILURE_TOTAL\", Failure_editor_name = _(\"AN/ALE-40(V) total failure\"),  Element = 10, Integrity_Treshold = 0.5, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("		{Failure = AN_ALE_40V_FAILURE_CONTAINER_LEFT_WING, Failure_name = \"AN_ALE_40V_FAILURE_CONTAINER_LEFT_WING\", Failure_editor_name = _(\"AN/ALE-40(V) left wing container failure\"),  Element = 23, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("		{Failure = AN_ALE_40V_FAILURE_CONTAINER_LEFT_GEAR, Failure_name = \"AN_ALE_40V_FAILURE_CONTAINER_LEFT_GEAR\", Failure_editor_name = _(\"AN/ALE-40(V) left gear container failure\"),  Element = 15, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("		{Failure = AN_ALE_40V_FAILURE_CONTAINER_RIGHT_GEAR, Failure_name = \"AN_ALE_40V_FAILURE_CONTAINER_RIGHT_GEAR\", Failure_editor_name = _(\"AN/ALE-40(V) right gear container failure\"),  Element = 16, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("		{Failure = AN_ALE_40V_FAILURE_CONTAINER_RIGHT_WING, Failure_name = \"AN_ALE_40V_FAILURE_CONTAINER_RIGHT_WING\", Failure_editor_name = _(\"AN/ALE-40(V) right wing container failure\"),  Element = 24, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("}\n")
    file:write("\n")
    file:write("need_to_be_closed = true -- lua_state  will be closed in post_initialize()\n")
    
    file:close()

    log('[DiCE] A10C2 CMDS Updated')
end


function writeA10CFile()
    log('[DiCE] Update A10C CMDS')
	
    local file = nil
    file = io.open(cmdsLua_A10C_fullPath, "w")
	
    file:write("local gettext = require(\"i_18n\")\n")
    file:write("_ = gettext.translate\n")
    file:write("\n")
    file:write("programs = {}\n")
    file:write("\n")
    file:write("-- Old generation radar SAM\n")
    file:write("programs['A'] = {}\n")
    file:write("programs['A'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramAChaff'] .. "\n")
    file:write("programs['A'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramAFlare'] .. "\n")
    file:write("programs['A'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramAInterval'] .. "\n")
    file:write("programs['A'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramACycle'] .. "\n")
    file:write("\n")
    file:write("-- Current generation radar SAM\n")
    file:write("programs['B'] = {}\n")
    file:write("programs['B'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramBChaff'] .. "\n")
    file:write("programs['B'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramBFlare'] .. "\n")
    file:write("programs['B'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramBInterval'] .. "\n")
    file:write("programs['B'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramBCycle'] .. "\n")
    file:write("\n")
    file:write("-- IR SAM\n")
    file:write("programs['C'] = {}\n")
    file:write("programs['C'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramCChaff'] .. "\n")
    file:write("programs['C'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramCFlare'] .. "\n")
    file:write("programs['C'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramCInterval'] .. "\n")
    file:write("programs['C'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramCCycle'] .. "\n")
    file:write("\n")
    file:write("-- Default manual presets\n")
    file:write("-- Mix 1\n")
    file:write("programs['D'] = {}\n")
    file:write("programs['D'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramDChaff'] .. "\n")
    file:write("programs['D'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramDFlare'] .. "\n")
    file:write("programs['D'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramDInterval'] .. "\n")
    file:write("programs['D'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramDCycle'] .. "\n")
    file:write("\n")
    file:write("-- Mix 2\n")
    file:write("programs['E'] = {}\n")
    file:write("programs['E'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramEChaff'] .. "\n")
    file:write("programs['E'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramEFlare'] .. "\n")
    file:write("programs['E'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramEInterval'] .. "\n")
    file:write("programs['E'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramECycle'] .. "\n")
    file:write("\n")
    file:write("-- Mix 3\n")
    file:write("programs['F'] = {}\n")
    file:write("programs['F'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramFChaff'] .. "\n")
    file:write("programs['F'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramFFlare'] .. "\n")
    file:write("programs['F'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramFInterval'] .. "\n")
    file:write("programs['F'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramFCycle'] .. "\n")
    file:write("\n")
    file:write("-- Mix 4\n")
    file:write("programs['G'] = {}\n")
    file:write("programs['G'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramGChaff'] .. "\n")
    file:write("programs['G'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramGFlare'] .. "\n")
    file:write("programs['G'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramGInterval'] .. "\n")
    file:write("programs['G'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramGCycle'] .. "\n")
    file:write("\n")
    file:write("-- Chaff single\n")
    file:write("programs['H'] = {}\n")
    file:write("programs['H'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramHChaff'] .. "\n")
    file:write("programs['H'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramHFlare'] .. "\n")
    file:write("programs['H'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramHInterval'] .. "\n")
    file:write("programs['H'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramHCycle'] .. "\n")
    file:write("\n")
    file:write("-- Chaff pair\n")
    file:write("programs['I'] = {}\n")
    file:write("programs['I'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramIChaff'] .. "\n")
    file:write("programs['I'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramIFlare'] .. "\n")
    file:write("programs['I'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramIInterval'] .. "\n")
    file:write("programs['I'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramICycle'] .. "\n")
    file:write("\n")
    file:write("-- Flare single\n")
    file:write("programs['J'] = {}\n")
    file:write("programs['J'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramJChaff'] .. "\n")
    file:write("programs['J'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramJFlare'] .. "\n")
    file:write("programs['J'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramJInterval'] .. "\n")
    file:write("programs['J'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramJCycle'] .. "\n")
    file:write("\n")
    file:write("-- Flare pair\n")
    file:write("programs['K'] = {}\n")
    file:write("programs['K'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramKChaff'] .. "\n")
    file:write("programs['K'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramKFlare'] .. "\n")
    file:write("programs['K'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramKInterval'] .. "\n")
    file:write("programs['K'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramKCycle'] .. "\n")
    file:write("\n")
    file:write("-- Chaff pre-empt\n")
    file:write("programs['L'] = {}\n")
    file:write("programs['L'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramLChaff'] .. "\n")
    file:write("programs['L'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramLFlare'] .. "\n")
    file:write("programs['L'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramLInterval'] .. "\n")
    file:write("programs['L'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramLCycle'] .. "\n")
    file:write("\n")
    file:write("-- Flare pre-empt\n")
    file:write("programs['M'] = {}\n")
    file:write("programs['M'][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramMChaff'] .. "\n")
    file:write("programs['M'][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramMFlare'] .. "\n")
    file:write("programs['M'][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramMInterval'] .. "\n")
    file:write("programs['M'][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE A-10C']['A10CProgramMCycle'] .. "\n")
    file:write("\n")
    file:write("\n")
    file:write("\n")
    file:write("ContainerChaffCapacity = 120\n")
    file:write("\n")
    file:write("ContainerFlareCapacity = 60\n")
    file:write("\n")
    file:write("NumberOfContiners      = 4\n")
    file:write("\n")
    file:write("AN_ALE_40V_FAILURE_TOTAL = 0\n")
    file:write("AN_ALE_40V_FAILURE_CONTAINER_LEFT_WING	= 1\n")
    file:write("AN_ALE_40V_FAILURE_CONTAINER_LEFT_GEAR	= 2\n")
    file:write("AN_ALE_40V_FAILURE_CONTAINER_RIGHT_GEAR	= 3\n")
    file:write("AN_ALE_40V_FAILURE_CONTAINER_RIGHT_WING	= 4\n")
    file:write("\n")
    file:write("Damage = {	{Failure = AN_ALE_40V_FAILURE_TOTAL, Failure_name = \"AN_ALE_40V_FAILURE_TOTAL\", Failure_editor_name = _(\"AN/ALE-40(V) total failure\"),  Element = 10, Integrity_Treshold = 0.5, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("		{Failure = AN_ALE_40V_FAILURE_CONTAINER_LEFT_WING, Failure_name = \"AN_ALE_40V_FAILURE_CONTAINER_LEFT_WING\", Failure_editor_name = _(\"AN/ALE-40(V) left wing container failure\"),  Element = 23, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("		{Failure = AN_ALE_40V_FAILURE_CONTAINER_LEFT_GEAR, Failure_name = \"AN_ALE_40V_FAILURE_CONTAINER_LEFT_GEAR\", Failure_editor_name = _(\"AN/ALE-40(V) left gear container failure\"),  Element = 15, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("		{Failure = AN_ALE_40V_FAILURE_CONTAINER_RIGHT_GEAR, Failure_name = \"AN_ALE_40V_FAILURE_CONTAINER_RIGHT_GEAR\", Failure_editor_name = _(\"AN/ALE-40(V) right gear container failure\"),  Element = 16, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("		{Failure = AN_ALE_40V_FAILURE_CONTAINER_RIGHT_WING, Failure_name = \"AN_ALE_40V_FAILURE_CONTAINER_RIGHT_WING\", Failure_editor_name = _(\"AN/ALE-40(V) right wing container failure\"),  Element = 24, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
    file:write("}\n")
    file:write("\n")
    file:write("need_to_be_closed = true -- lua_state  will be closed in post_initialize()\n")
    
    file:close()

    log('[DiCE] A10C CMDS Updated')
end


function writeF16CFile() -- BYPASS is not included
    log('[DiCE] Update F16C CMDS')
	
    local file = nil
    file = io.open(cmdsLua_F16C_fullPath, "w")
        
	file:write("local gettext = require(\"i_18n\")\n")
	file:write("_ = gettext.translate\n")
	file:write("\n")
	file:write("local count = 0\n")
	file:write("local function counter()\n")
	file:write("	count = count + 1\n")
	file:write("	return count\n")
	file:write("end\n")
	file:write("\n")
	file:write("ProgramNames =\n")
	file:write("{\n")
	file:write("	MAN_1 = counter(),\n")
	file:write("	MAN_2 = counter(),\n")
	file:write("	MAN_3 = counter(),\n")
	file:write("	MAN_4 = counter(),\n")
	file:write("	MAN_5 = counter(),\n")
	file:write("	MAN_6 = counter(),\n")
	file:write("	AUTO_1 = counter(),\n")
	file:write("	AUTO_2 = counter(),\n")
	file:write("	AUTO_3 = counter(),\n")
	file:write("	BYP = 27,\n")
	file:write("}\n")
	file:write("\n")
	file:write("programs = {}\n")
	file:write("\n")
	file:write("-- Default manual presets\n")
	file:write("-- MAN 1\n")
	file:write("programs[ProgramNames.MAN_1] = {\n")
	file:write("	chaff = {\n")
	file:write("		burstQty 	= " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual1ChaffBurstQty'] .. ",\n")
	file:write("		burstIntv	= " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual1ChaffBurstIntv'] .. ",\n")
	file:write("		salvoQty	= " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual1ChaffSalvoQty'] .. ",\n")
	file:write("		salvoIntv	= " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual1ChaffSalvoIntv'] .. ",\n")
	file:write("	},\n")
	file:write("	flare = {\n")
	file:write("		burstQty	= " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual1FlareBurstQty'] .. ",\n")
	file:write("		burstIntv	= " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual1FlareBurstIntv'] .. ",\n")
	file:write("		salvoQty	= " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual1FlareSalvoQty'] .. ",\n")
	file:write("		salvoIntv	= " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual1FlareSalvoIntv'] .. ",\n")
	file:write("	},\n")
	file:write("}\n")
	file:write("\n")
	file:write("-- MAN 2\n")
	file:write("programs[ProgramNames.MAN_2] = {\n")
	file:write("   chaff = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual2ChaffBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual2ChaffBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual2ChaffSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual2ChaffSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("    flare = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual2FlareBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual2FlareBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual2FlareSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual2FlareSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("}\n")
    file:write("\n")
	file:write("-- MAN 3\n")
	file:write("programs[ProgramNames.MAN_3] = {\n")
	file:write("   chaff = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual3ChaffBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual3ChaffBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual3ChaffSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual3ChaffSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("    flare = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual3FlareBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual3FlareBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual3FlareSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual3FlareSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("}\n")
    file:write("\n")
	file:write("-- MAN 4\n")
	file:write("programs[ProgramNames.MAN_4] = {\n")
	file:write("   chaff = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual4ChaffBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual4ChaffBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual4ChaffSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual4ChaffSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("    flare = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual4FlareBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual4FlareBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual4FlareSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual4FlareSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("}\n")
    file:write("\n")
	file:write("-- MAN 5 - Wall Dispense button, Panic\n")
	file:write("programs[ProgramNames.MAN_5] = {\n")
	file:write("   chaff = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual5ChaffBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual5ChaffBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual5ChaffSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual5ChaffSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("    flare = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual5FlareBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual5FlareBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual5FlareSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual5FlareSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("}\n")
    file:write("\n")
	file:write("-- MAN 6 - CMS LEFT\n")
	file:write("programs[ProgramNames.MAN_6] = {\n")
	file:write("   chaff = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual6ChaffBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual6ChaffBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual6ChaffSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual6ChaffSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("    flare = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual6FlareBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual6FlareBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual6FlareSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CManual6FlareSalvoIntv'] .. ",\n")
    file:write("    },\n")
    file:write("}\n")
    file:write("\n")
	file:write("-- Auto presets\n")
	file:write("-- Threat type 1\n")
	file:write("programs[ProgramNames.AUTO_1] = {\n")
	file:write("	chaff = {\n")
	file:write("		burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto1ChaffBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto1ChaffBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto1ChaffSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto1ChaffSalvoIntv'] .. ",\n")
	file:write("	},\n")
	file:write("	flare = {\n")
	file:write("		burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto1FlareBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto1FlareBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto1FlareSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto1FlareSalvoIntv'] .. ",\n")
	file:write("	},\n")
	file:write("}\n")
	file:write("\n")
	file:write("-- Threat type 2\n")
	file:write("programs[ProgramNames.AUTO_2] = {\n")
	file:write("	chaff = {\n")
	file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto2ChaffBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto2ChaffBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto2ChaffSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto2ChaffSalvoIntv'] .. ",\n")
    file:write("	},\n")
	file:write("	flare = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto2FlareBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto2FlareBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto2FlareSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto2FlareSalvoIntv'] .. ",\n")
	file:write("	},\n")
	file:write("}\n")
	file:write("\n")
	file:write("-- Threat type 3\n")
	file:write("programs[ProgramNames.AUTO_3] = {\n")
	file:write("	chaff = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto3ChaffBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto3ChaffBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto3ChaffSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto3ChaffSalvoIntv'] .. ",\n")
	file:write("	},\n")
	file:write("	flare = {\n")
    file:write("        burstQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto3FlareBurstQty'] .. ",\n")
    file:write("        burstIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto3FlareBurstIntv'] .. ",\n")
    file:write("        salvoQty    = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto3FlareSalvoQty'] .. ",\n")
    file:write("        salvoIntv   = " .. optionsLuaPathData['options']['plugins']['DiCE F-16C']['F16CAuto3FlareSalvoIntv'] .. ",\n")
	file:write("	},\n")
	file:write("}\n")
	file:write("\n")
	file:write("-- BYPASS\n")
	file:write("programs[ProgramNames.BYP] = {\n")
	file:write("	chaff = {\n")
	file:write("		burstQty 	= 1,\n")
	file:write("		burstIntv	= 0.02,\n")
	file:write("		salvoQty	= 1,\n")
	file:write("		salvoIntv	= 0.5,\n")
	file:write("	},\n")
	file:write("	flare = {\n")
	file:write("		burstQty	= 1,\n")
	file:write("		burstIntv	= 0.02,\n")
	file:write("		salvoQty	= 1,\n")
	file:write("		salvoIntv	= 0.5,\n")
	file:write("	},\n")
	file:write("}\n")
	file:write("\n")
	file:write("AN_ALE_47_FAILURE_TOTAL = 0\n")
	file:write("AN_ALE_47_FAILURE_CONTAINER	= 1\n")
	file:write("\n")
	file:write("Damage = {	{Failure = AN_ALE_47_FAILURE_TOTAL, Failure_name = \"AN_ALE_47_FAILURE_TOTAL\", Failure_editor_name = _(\"AN/ALE-47 total failure\"),  Element = 10, Integrity_Treshold = 0.5, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
	file:write("			{Failure = AN_ALE_47_FAILURE_CONTAINER, Failure_name = \"AN_ALE_47_FAILURE_CONTAINER\", Failure_editor_name = _(\"AN/ALE-47 container failure\"),  Element = 23, Integrity_Treshold = 0.75, work_time_to_fail_probability = 0.5, work_time_to_fail = 3600*300},\n")
	file:write("}\n")
	file:write("\n")
	file:write("need_to_be_closed = true -- lua_state  will be closed in post_initialize()\n")
        
    file:close()

    log('[DiCE] F16C CMDS Updated')
end


function writeF18CFile()
	log('[DiCE] Update F18 CMDS')	
	
	local file = nil
	--https://www.tutorialspoint.com/lua/lua_file_io.htm
	file = io.open(cmdsLua_F18C_fullPath, "w") -- open the file to prepare for a write
	
	-- start writing line by line.
	-- If dcs updates to a new format then this
	-- will be out of date. 
		
	file:write("local count = 0\n")
	file:write("local function counter()\n")
	file:write("	count = count + 1\n")
	file:write("	return count\n")
	file:write("end\n")
	file:write("\n")
	file:write("ProgramNames =\n")
	file:write("{\n")
	file:write("	MAN_1 = counter(),\n")
	file:write("	MAN_2 = counter(),\n")
	file:write("	MAN_3 = counter(),\n")
	file:write("	MAN_4 = counter(),\n")
	file:write("	MAN_5 = counter(),\n")
	file:write("	MAN_6 = counter(),\n")
	file:write("	AUTO_1 = counter(),\n")
	file:write("	AUTO_2 = counter(),\n")
	file:write("	AUTO_3 = counter(),\n")
	file:write("	AUTO_4 = counter(),\n")
	file:write("	AUTO_5 = counter(),\n")
	file:write("	AUTO_6 = counter()\n")
	file:write("}\n")
	file:write("	\n")
	file:write("\n")
	file:write("programs = {}\n")
	file:write("\n")
	file:write("-- Default manual presets\n")
	file:write("-- MAN 1\n")
	file:write("programs[ProgramNames.MAN_1] = {}\n")
	file:write("programs[ProgramNames.MAN_1][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual1Chaff'] .. "\n")
	file:write("programs[ProgramNames.MAN_1][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual1Flare'] .. "\n")
	file:write("programs[ProgramNames.MAN_1][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual1Interval'] .. "\n")
	file:write("programs[ProgramNames.MAN_1][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual1Cycle'] .. "\n")
	file:write("\n")
	file:write("-- MAN 2\n")
	file:write("programs[ProgramNames.MAN_2] = {}\n")
	file:write("programs[ProgramNames.MAN_2][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual2Chaff'] .. "\n")
	file:write("programs[ProgramNames.MAN_2][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual2Flare'] .. "\n")
	file:write("programs[ProgramNames.MAN_2][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual2Interval'] .. "\n")
	file:write("programs[ProgramNames.MAN_2][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual2Cycle'] .. "\n")
	file:write("\n")
	file:write("-- MAN 3\n")
	file:write("programs[ProgramNames.MAN_3] = {}\n")
	file:write("programs[ProgramNames.MAN_3][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual3Chaff'] .. "\n")
	file:write("programs[ProgramNames.MAN_3][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual3Flare'] .. "\n")
	file:write("programs[ProgramNames.MAN_3][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual3Interval'] .. "\n")
	file:write("programs[ProgramNames.MAN_3][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual3Cycle'] .. "\n")
	file:write("\n")
	file:write("-- MAN 4\n")
	file:write("programs[ProgramNames.MAN_4] = {}\n")
	file:write("programs[ProgramNames.MAN_4][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual4Chaff'] .. "\n")
	file:write("programs[ProgramNames.MAN_4][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual4Flare'] .. "\n")
	file:write("programs[ProgramNames.MAN_4][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual4Interval'] .. "\n")
	file:write("programs[ProgramNames.MAN_4][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual4Cycle'] .. "\n")
	file:write("\n")
	file:write("-- MAN 5 - Chaff single\n")
	file:write("programs[ProgramNames.MAN_5] = {}\n")
	file:write("programs[ProgramNames.MAN_5][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual5Chaff'] .. "\n")
	file:write("programs[ProgramNames.MAN_5][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual5Flare'] .. "\n")
	file:write("programs[ProgramNames.MAN_5][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual5Interval'] .. "\n")
	file:write("programs[ProgramNames.MAN_5][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual5Cycle'] .. "\n")
	file:write("\n")
	file:write("-- MAN 6 - Wall Dispense button, Panic\n")
	file:write("programs[ProgramNames.MAN_6] = {}\n")
	file:write("programs[ProgramNames.MAN_6][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual6Chaff'] .. "\n")
	file:write("programs[ProgramNames.MAN_6][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual6Flare'] .. "\n")
	file:write("programs[ProgramNames.MAN_6][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual6Interval'] .. "\n")
	file:write("programs[ProgramNames.MAN_6][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CManual6Cycle'] .. "\n")
	file:write("\n")
	file:write("-- Auto presets\n")
	file:write("-- Threat type 1\n")
	file:write("programs[ProgramNames.AUTO_1] = {}\n")
	file:write("programs[ProgramNames.AUTO_1][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto1Chaff'] .. "\n")
	file:write("programs[ProgramNames.AUTO_1][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto1Flare'] .. "\n")
	file:write("programs[ProgramNames.AUTO_1][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto1Interval'] .. "\n")
	file:write("programs[ProgramNames.AUTO_1][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto1Cycle'] .. "\n")
	file:write("\n")
	file:write("-- Threat type 2\n")
	file:write("programs[ProgramNames.AUTO_2] = {}\n")
	file:write("programs[ProgramNames.AUTO_2][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto2Chaff'] .. "\n")
	file:write("programs[ProgramNames.AUTO_2][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto2Flare'] .. "\n")
	file:write("programs[ProgramNames.AUTO_2][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto2Interval'] .. "\n")
	file:write("programs[ProgramNames.AUTO_2][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto2Cycle'] .. "\n")
	file:write("\n")
	file:write("-- Threat type 3\n")
	file:write("programs[ProgramNames.AUTO_3] = {}\n")
	file:write("programs[ProgramNames.AUTO_3][\"chaff\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto3Chaff'] .. "\n")
	file:write("programs[ProgramNames.AUTO_3][\"flare\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto3Flare'] .. "\n")
	file:write("programs[ProgramNames.AUTO_3][\"intv\"]  = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto3Interval'] .. "\n")
	file:write("programs[ProgramNames.AUTO_3][\"cycle\"] = " .. optionsLuaPathData['options']['plugins']['DiCE F-18C']['F18CAuto3Cycle'] .. "\n")
	file:write("\n")
	file:write("\n")
	file:write("need_to_be_closed = true -- lua_state  will be closed in post_initialize()\n")
	
	file:close() 

	log('[DiCE] F18 CMDS Updated')
end


function file_exists(name)
   local f=io.open(name,"r")
   if f~=nil then io.close(f) return true else return false end
end

end -- end function loadDiCE()


local status, err = pcall(loadDiCE)
if not status then
    net.log("[DiCE] Load Error: " .. tostring(err))
end