local base = _G

module('me_map_window')

local Window = base.require('Window')
local MapView = base.require('MapView')
local MapState = base.require('MapState')
local MsgWindow = base.require('MsgWindow')
local U = base.require('me_utilities')
local BScolor = base.require('BScolor')
local Color = base.require('Color')
local Mission = base.require('me_mission')
local Roads = base.require('Roads')
local Terrain = base.require('edTerrain')
local Theme = base.require('Theme')
local db = base.require('me_db_api')
local S = base.require('Serializer')
local i18n = base.require('i18n')
local crutches = base.require('me_crutches')  -- temporary crutches



i18n.setup(_M)

cdata = {
    farp = _('FARP');
    placeInside = _('Please place objects into map');
}

initialAltitude = {
    helicopter = 500,
    plane = 2000,
}

initialVelocity = {
    helicopter = 200/3.6,
    plane = 500/3.6,
}

xPrev = 0
yPrev = 0
moving = false
newAirGroupNumber = 0
MOUSE_POS = {
    pressTime = base.os.clock();
}

-------------------------------------------------------------------------------
function initModule()
    waypointDragged  = false;
end;

-------------------------------------------------------------------------------
function load()
    base.Roads.Create(44, 72)
    base.Roads.LoadBin(base.mainPath..'data/map/roads.bin')
end;

-------------------------------------------------------------------------------
function create(x, y, w, h)
  startParameters = {x, y, w, h}
end

-------------------------------------------------------------------------------
function doCreate(x, y, w, h)
  window = Window.new(x, y, w, h)
  window:setTitleHeight(0)
  -- function window:onKeyDown(key, char)
    -- Window.onKeyDown(self, key, char)
  -- end
  mapView = MapView.new()
  panState = MapState.new(mapView)
  creatingPlaneState = MapState.new(mapView)
  creatingHelicopterState = MapState.new(mapView)
  creatingShipState = MapState.new(mapView)
  creatingVehicleState = MapState.new(mapView)
  creatingStaticState = MapState.new(mapView)
  addingWaypointState = MapState.new(mapView)
  addingTargetState = MapState.new(mapView)
  creatingZoneState = MapState.new(mapView)
  creatingTemplateState = MapState.new(mapView)
  creatingINUFixPointState = MapState.new(mapView)
  tapeState = MapState.new(mapView)
--  selState = MapState.new(map)
  mapView:setState(panState)
  map_w = w
  map_h = h

  mapView:setBounds(0, 0, w, h)
  window:addChild(mapView)

  setCallbacks()

  -- Порядок загрузки важен!!!
  -- 1
  mapView:loadClassifier(base.mainPath .. 'data/map/classifier.lua')
  -- 2
  local options = base.panel_options.getOptions()
  local images = options.difficulty.iconsTheme
  if not images then
      images = 'russian'
  end
  images = base.mainPath .. 'data/map/images/themes/' .. images .. '/'
  mapView:loadClassifierView(base.mainPath .. 'data/map/ClassifierView.lua', 
        images, Theme.fontsPath)
  -- 3
  mapView:loadTitles(i18n.getLocalizedFileName(base.mainPath .. 'data/map/titles.lua'))  
  -- 4 
  mapView:setBkgFolder(base.mainPath .. 'data/map/')
  -- 5
  mapView:loadLists(base.mainPath .. 'data/map/map.zip')
  -- 6
  for i, layer in base.ipairs(getLayers()) do 
    if layer.key == 'BUILDINGS' then
      if  base.panel_map_options.vdata.layers[i] then
        base.print('starting to load buildings');
        mapView:loadBuildings(base.mainPath .. 'data/map/buildings.zip')
        buildingsLoaded = true;
      else
        buildingsLoaded = false;
      end;
    end;
  end;

  -- Листы карты загружаются в BlackShark.lua
  scales = mapView:getScales()
  mapView:setCamera(0.7425065375, 0.7365236411) -- позиционируемся на границу Абхазии и Грузии
  mapView:setScale(1000000)  -- 1:1000000

    if not base.panel_map_options.created then
        base.panel_map_options.doCreate()
    end;
    base.panel_map_options.setupMap()
    created = true;
end;

-------------------------------------------------------------------------------
function getObjectById(id)
    return base.module_mission.mapObjects[id]
end

-------------------------------------------------------------------------------
-- возвращает тип поверхности
-- lat, lon - координаты в радианах
-- radius - радиус в радианах
-- возвращает:
-- 'sea' - море
-- 'lake' - озеро
-- 'river' - река
-- 'land' - земля
function getSurfaceType(lat, lon, radius)
  local result = 'land'
  
  -- в objects classKey объектов расположены также, как объекты видны на карте, 
  -- то есть первыми идут classKey объектов с верхних слоев карты
  local objects = mapView:findObjects(lat, lon, radius)
  
  for _tmp, classKey in base.pairs(objects) do
    -- острова расположены над водой
    if 'P000000800' == classKey then
      result = 'land'
      break
    end  
    
    -- ОКЕАНЫ И МОРЯ
    if 'S0031110000' == classKey or 'S00311100001' == classKey then
      result = 'sea'
      break
    end  
    
    -- ОЗЕРА
    if 'S0031120000' == classKey or 'S00311200001' == classKey then
      result = 'lake'
      break      
    end
    
    -- реки
    if 'S00314100001' == classKey or 'S0031410000' == classKey or
       'L00314100003' == classKey or 'L0031410000' == classKey then       
      result = 'river'
      break
    end
  end
  
  return result
end

-------------------------------------------------------------------------------
function showWarningWindow(text)
  local msgWindow = MsgWindow.new(text, ' Warning', 'warning', 'OK')
  msgWindow:setVisible(true)  
end

-------------------------------------------------------------------------------
-- Фильтр задает список classKey выбираемых объектов в виде: {["classKey"]=true,...}.
-- Возвращает первый попавшийся пользовательский объект.
-- Возвращаются не только точечные, но и линейные объекты!
function pickUserObject(x, y, radius, selectNext)
    local objects = mapView:findUserObjects(x, y, radius) -- возвращает таблицу вида {id, id, ...}
    local cx,cy = MapView:convertRadiansToMeters( x, y );
    if nil == cx then
        return nil
    end
  
  
    local dist = base.math.huge; -- макс. число
    local index = 0; -- индекс объекта в массиве, который ближе всего к клику
    local firstObj; -- первый объект  списка
    local allowSelection = true;
    local objectsCount = 0;
    local selectedObjectPresent = false;
    for i = 1, #objects do
      local obj = getObjectById(objects[i])
      --base.print('obj', i, 'picked');
      --base.U.traverseTable(obj, 2);
      local classKey = obj.classKey
      if 'P' == base.string.sub(classKey, 1, 1) then
          firstObj = firstObj or obj; -- инициализация первого объекта
          if "DOT" == obj.type then 
            if obj.id  == selectedObject then
                selectedObjectPresent = true;
            end;
              objectsCount = objectsCount + 1;
              local lat, long = obj.points[1][1].x, obj.points[1][1].y
              local x,y = MapView:convertRadiansToMeters( lat, long );
              local d = base.math.sqrt( (x - cx)^2 + (y - cy)^2 ); -- расстояние до точки клика
              if d < dist then
                  index = i; -- запоминаем индекс
              end;
          end; 
      end
    end

    local nearestObject;
    if index > 0 then -- если нашли ближайший объект
      nearestObject =  getObjectById(objects[index]); 
    else 
      nearestObject =  firstObj; -- возвращаем первый объект
    end;
    
    -- если в радиусе только один объект или нет никого, то выделяем обычным способом
    if objectsCount <= 1 then
        return nearestObject, true;
    end;

    -- если объектов несколько, то смотрим выделенный объект
    if selectedObjectPresent then 
        if true == selectNext then
            -- хотим выделить следующий объект
            return firstObj, true;
        else
            -- не хотим выделять объектов
            return nearestObject, false;
        end;
    else -- выделенного объекта нет, выделяем ближайший
        return nearestObject, true;
    end;
    
end


-------------------------------------------------------------------------------
-- remove selection
function unselectAll()
    if selectedGroup then
        local g = selectedGroup;
        revert_selection(selectedGroup)        
        selectedGroup = nil;
        if g then
            base.module_mission.update_group_map_objects(g);
        end;
        base.panel_fix_points.vdata.selectedPoint = nil;
    end
    if base.panel_zone.currentZone then
        base.panel_zone.currentZone = nil
    end
end

-------------------------------------------------------------------------------
-- Восстанавливаются умалчиваемые цвета объектов группы на карте.
function revert_selection(group)
    if not group.mapObjects then
        return;
    end;
  for i,v in base.pairs(group.mapObjects.units) do
    v.currColor = group.color
  end
  if group.mapObjects.route then
    for i,v in base.pairs(group.mapObjects.route.points) do
      v.currColor = group.color
    end
    for i,v in base.pairs(group.mapObjects.route.numbers) do
      v.currColor = group.color
    end
    group.mapObjects.route.line.currColor = group.color
    for i,v in base.pairs(group.mapObjects.route.targets) do
      for j,u in base.pairs(v) do
        u.currColor = group.color
      end
    end
    for i,v in base.pairs(group.mapObjects.route.targetNumbers) do
      for j,u in base.pairs(v) do
        u.currColor = group.color
      end
    end
    for i,v in base.pairs(group.mapObjects.route.targetLines) do
      for j,u in base.pairs(v) do
        u.currColor = group.color
      end
    end
  end
  
  if group.mapObjects.INUFixPoints then
    for i,v in base.pairs(group.mapObjects.INUFixPoints) do
        v.currColor = group.boss.boss.selectGroupColor;
    end;
  end;
  if group.mapObjects.INUFixPoints_numbers then
    for i,v in base.pairs(group.mapObjects.INUFixPoints_numbers) do
        v.currColor = group.boss.boss.selectGroupColor;
    end;
  end;
  base.module_mission.update_group_map_objects(group)
  mapView:updateUserList(true)
end

-------------------------------------------------------------------------------
-- Устанавливается цвет объектов карты для выбранной группы.
function set_selected_group_color(group)
  local select_color = group.boss.boss.selectGroupColor;
  for i,v in base.pairs(group.mapObjects.units) do
    v.currColor = select_color
  end
  
  if group.mapObjects.route then
    for i,v in base.pairs(group.mapObjects.route.points) do
      v.currColor = select_color
    end

    for i,v in base.pairs(group.mapObjects.route.numbers) do
      v.currColor = select_color
    end
    
    group.mapObjects.route.line.currColor = select_color  
    
    for i,v in base.pairs(group.mapObjects.route.targets) do
        for k,w in base.pairs(v) do
            w.currColor = select_color
        end;
    end
    for i,v in base.pairs(group.mapObjects.route.targetZones) do
        for k,w in base.pairs(v) do
            w.currColor[1] = select_color[1]
            w.currColor[2] = select_color[2]
            w.currColor[3] = select_color[3]
        end;
    end
    for i,v in base.pairs(group.mapObjects.route.targetNumbers) do
        for k,w in base.pairs(v) do
            w.currColor = select_color
        end;
    end
    for i,v in base.pairs(group.mapObjects.route.targetLines) do
        for k,w in base.pairs(v) do
            w.currColor = select_color
        end;
    end
  end;
    
  base.module_mission.update_group_map_objects(group)
  mapView:updateUserList(true)
end

-------------------------------------------------------------------------------
-- start show ruler
function startRuler(x, y)
    if tape then
        Mission.mapObjects[tape.id] = nil
        Mission.mapObjects[tape_text.id] = nil
        mapView:removeUserObjects({tape, tape_text})
    end
    local mapX, mapY = mapView:getMapPoint(x, y)
    local scale = mapView:getScale()
    local coeff = scale/100000
    -- Устанавливается начальная точка линейки.
    -- Создаются отрезок и подпись.
    Mission.currentKey = Mission.currentKey + 1
    tape = {
        type = 'LIN',
        key = Mission.currentKey,
        id = Mission.currentKey,
        classKey = 'L0000000525',
        points = {
            [1] = {
                [1] = {x=mapX, y=mapY},
                [2] = {x=mapX, y=mapY},
            },
        },
    }
    Mission.mapObjects[tape.id] = tape
    Mission.currentKey = Mission.currentKey + 1
    local text = '0 m, 0'
    tape_text = {
        type = 'TIT',
        key = Mission.currentKey,
        id = Mission.currentKey,
        classKey = 'T0000000527',
        points = {
            [1] = {
                [1] = {x=mapX+0.00005*coeff, y=mapY+0.0001*coeff},
                [2] = {x=mapX+0.00005*coeff, y=mapY+0.0011*coeff},
            },
        },
        title = text,
        semantics = {
            [1] = {code=9, value = text,},
            [2] = {code=14, value = '5',},
            [3] = {code=94, value = '301',},
        },
    }
    Mission.mapObjects[tape_text.id] = tape
    mapView:addUserObjects({tape, tape_text})
    mapView:updateUserList(true)
end

-------------------------------------------------------------------------------
-- called on ruler extended
function moveRuler(dx, dy, x, y)
    if tape then
        local mapX, mapY = mapView:getMapPoint(x, y)
        local scale = mapView:getScale()
        local coeff = scale/100000
        -- Устанавливается конечная точка линейки.
        tape.points[1][2] = {x=mapX, y=mapY}
        base.print( mapX, mapY)
        -- Обновляется подпись.
        --local x1,z1 = Roads.xz(tape.points[1][1].x, tape.points[1][1].y)
        --local x2,z2 = Roads.xz(tape.points[1][2].x, tape.points[1][2].y)
        --local dist = base.math.floor(base.math.sqrt((x1-x2)*(x1-x2)+(z1-z2)*(z1-z2))+0.5)
        local lat1 = tape.points[1][1].x
        local long1 = tape.points[1][1].y
        local lat2 = tape.points[1][2].x
        local long2 = tape.points[1][2].y
        local dist = base.math.floor(db.getDist(lat1, long1, lat2, long2))
        --local ang = base.math.mod(base.math.floor(360+90-base.math.atan2(lat2-lat1, long2-long1)*180/base.math.pi), 360)
        local cos = base.math.cos;
        local sin = base.math.sin;
        local atan2 = base.math.atan2;
        local pi  = base.math.pi;
        
        local dl = long2 - long1;
        ang = atan2(sin(dl)*cos(lat2), cos(lat1)*sin(lat2) - sin(lat1)*cos(lat2)*cos(dl));		
        ang = base.math.floor((ang * 180/pi)*10)/10;
		
		if (ang < 0) then
			ang = 360 + ang
		end
		
        local text = dist ..' m, '..ang..'°'
--        local text = dist ..' m, '..ang..'°' .. ' dlat ' .. (lat2 - lat1) .. ' dlon ' .. (long2 - long1)
        tape_text.title = text
        tape_text.semantics[1].value = text
        tape_text.points[1][1] = {x=mapX+0.00005*coeff, y=mapY+0.0001*coeff}
        tape_text.points[1][2] = {x=mapX+0.00005*coeff, y=mapY+0.0011*coeff}
        mapView:removeUserObjects({tape, tape_text})
        mapView:addUserObjects({tape, tape_text})
        mapView:updateUserList(true)
    end
end

-------------------------------------------------------------------------------
-- end with ruler
function finishRuler(x, y)
    if tape then
        Mission.mapObjects[tape.id] = nil
        Mission.mapObjects[tape_text.id] = nil
        mapView:removeUserObjects({tape, tape_text})
        mapView:updateUserList(true)
        tape = nil
    end
end

-------------------------------------------------------------------------------
-- Нужно четко разграничить состояния карты, чтобы при любом переключении диалогов
-- соответствующим образом переключались и состояния карты.
-- Например, состояние panState предназначено только для обзора карты и выборки объектов миссии.
-- В этом состоянии нельзя создавать и редактировать объекты миссии на карте.
-- Когда, например, нажимается одна из кнопок групп юнитов, карта переводится в режим
-- создания группы соответствующего типа (createPlaneGroupState, createShipGroupState и т.п.) до тех пор,
-- пока группа не будет создана.
-- После создания группы карта переводится в режим добавления точки маршрута addWaypointState.
-- В этом режиме также могут вводиться состояния редактирования точки маршрута editWaypointState,
-- добавления цели в точке маршрута addWaypointTarget и т.д.
-- Наконец, при закрытии диалога группы карта вновь переводится в умалчиваемое состояние panState.
-- Таким образом, у разработчика всегда имеется возможность при переключении диалогов сбросить
-- специфическое состояние карты и перевести его в умалчиваемое вызовом функции mapView:setState(panState).
function hideGroupPanels()
    base.panel_aircraft.show(false)
    base.panel_static.show(false)
    base.panel_ship.show(false)
    base.panel_vehicle.show(false)
    base.panel_route.show(false)
    base.panel_targeting.show(false)
    base.panel_payload.show(false)
    base.panel_summary.show(false)
    base.panel_fix_points.show(false)
    base.panel_trig_zones_list.show(false)
    base.panel_zone.show(false)
        
end

-------------------------------------------------------------------------------
-- Функции умалчиваемого режима карты PAN.
function panState_onMouseUp(self, x, y, button)
    MOUSE_POS = MOUSE_POS or {};
    local drag = MOUSE_POS.drag;  -- показывает таскали ли мы объект
                -- необходимо для того, чтобы предотвратить ситуацию, когда
                -- объект схватили, потаскали, а потом поставили в первоначальную точку
    MOUSE_POS.drag = false;
    if 2 == button then
      finishRuler(x, y)
      return
    elseif (4 == button) or (5 == button) then
    else
        if MOUSE_POS.x and MOUSE_POS.y then			
            if base.math.sqrt( (MOUSE_POS.x - x)^2 + (MOUSE_POS.y - y)^2 ) <= 2 then 
                -- позиции нажатия и отпускания совпали, надо выбрать следующий объект
                if true ~= drag then
                    handleLeftMouseDown(MOUSE_POS.x, MOUSE_POS.y, true);
                    MOUSE_POS.handled = true;
                end;
            end;
        end;
        if waypointDragged then
            local group = selectedGroup
            local wpt = waypointDragged
            local alt = U.getAltitude(wpt.lat, wpt.long)
            if group ~= nil then
                if group.type == 'plane' or group.type == 'helicopter' then
                    alt = base.math.max(alt, wpt.alt)
                end
            end;
            wpt.alt = alt

            if base.panel_route.isAirfieldWaypoint(waypointDragged.type) then
                base.panel_route.attractToAirfield(waypointDragged,
                        selectedGroup)
            end
            local rvd = base.panel_route.vdata;
			
            if rvd and rvd.wpt and rvd.wpt.type and ('On Road' == rvd.wpt.type.action) then
                base.module_mission.move_waypoint_to_road(rvd.wpt);
                local group = rvd.wpt.boss;
                for i = 2, #group.units do
                    local unit = group.units[i];
                    base.module_mission.move_unit_to_road(unit);
                end;
            end;
          
			if (group) then
				base.module_mission.calc_route_length(group)
			end
			
            base.panel_summary.update()
            base.panel_route.update()
        end
        if selectedGroup and selectedUnit and selectedGroup.route and  selectedGroup.route.points  then 
            if 'On Road' == selectedGroup.route.points[1].type.action then
                --base.U.traverseTable(selectedGroup,3);
                --base.U.traverseTable(selectedUnit,3);
                base.module_mission.move_unit_to_road(selectedUnit);
            end;
        end;
    end
    waypointDragged = nil
    MapState.onMouseUp(self, x, y, button)
  end

-------------------------------------------------------------------------------
function handleLeftMouseDown(x, y, selectNext)
    if true == MOUSE_POS.handled then 		
        return
    end;
    local scale = mapView:getScale()
    local mapX, mapY = mapView:getMapPoint(x, y)
    -- Либо выбираем существующую группу, либо сбрасываем выбор.
    -- Сначала нужно восстановить родные цвета ранее выбранной группы.
    local iconSize = 10
    local rx, ry = mapView:getMapPoint(x + iconSize, y)
    local radius = ry - mapY
    local obj, allowSelection = pickUserObject(mapX, mapY, radius, selectNext)
    MOUSE_POS.lastPickedObject = obj;

    if allowSelection   then -- объект один или никого нет, действуем по обычной процедуре
        MOUSE_POS.handled = true;
    else --объектов несколько, действия производим в момент отпускания мыши
        MOUSE_POS.handled = false;
        return;
    end;
    waypointDragged = nil
    unselectAll()
    -- Радиус зоны выборки зависит от текущего масштаба.
    -- Здесь он подобран под размеры иконок юнитов и точек маршрута.
    -- радиус стоит сделать равным либо 1/2 размера иконок - радиус вписаной окружности,
    -- либо равным 1/2 размера иконок * 2^0.5 - радиус описанной окружности
    if obj and obj.userObject then
        --base.print('obj, obj.userObject, obj.userObject.boss, obj.type',obj, obj.userObject, obj.userObject.boss, obj.type);
        if not obj.userObject.boss and obj.classKey == 'P0000000529' then
            -- Центр зоны триггера
            selectedObject = obj.id
            panel = base.panel_zone
            panel.currentZone = obj.userObject
            if not panel.window:isVisible() then
                base.toolbar.untoggle_all_except()
            end
            panel.setSafeMode(false)
            panel.show(true)
            base.panel_trig_zones_list.selectZone(panel.currentZone)
        elseif obj.userObject.boss and (obj.type == 'DOT') then
            -- Теперь поработаем с выбранной группой.
            --        base.print(obj.classKey)
            --base.U.traverseTable(obj.userObject.boss, 3);
            selectedObject = obj.id
            local group = obj.userObject.boss
            if obj.classKey == 'P0091000044' then
                group = group.boss
            end
            local anchorType
            selectedUnit = nil
            if obj.waypoint then
                anchorType = 'waypoint'
            elseif 'unit' == obj.subclass then
                anchorType = 'unit'
                selectedUnit = obj.userObject
            end
            if obj.classKey == 'P0091000206' then -- INU fix point
                --base.print('INU fix point clicked')
            end;
            respondToSelectedUnit(obj, group, selectedUnit)
            --if base.panel_units_list.window:isVisible() then
            base.panel_units_list.selectRow(group);
            --end;
        else
            hideGroupPanels()
            base.toolbar.untoggle_all_except()
        end
    else
        hideGroupPanels()
        base.panel_units_list.show(false);
        --base.toolbar.untoggle_all_except()
    end
    
end;   
  
-------------------------------------------------------------------------------
function ZoomIn(x, y)
    local lat0, lon0 = mapView:getCamera()
    local lat, lon = mapView:getMapPoint(x, y)
    zoomIn()
    local lat1, lon1 = mapView:getMapPoint(x, y)
    mapView:setCamera(lat0+lat-lat1, lon0+lon-lon1)
    mapView:updateUserList(false)    
end;   

-------------------------------------------------------------------------------
function ZoomOut(x, y)
    local lat0, lon0 = mapView:getCamera()
    local lat, lon = mapView:getMapPoint(x, y)
    zoomOut()
    local lat1, lon1 = mapView:getMapPoint(x, y)
    mapView:setCamera(lat0+lat-lat1, lon0+lon-lon1)
    mapView:updateUserList(false)
end;  

-------------------------------------------------------------------------------
function panState_onMouseDown(self, x, y, button)
    --base.debug.sethook ( base.hookFcn, 'c' );
    -- Нажатие левой кнопки мыши на объекте группы вызывает его выбор без перемещения.
    MOUSE_POS = MOUSE_POS or {};
    MOUSE_POS.x = x;
    MOUSE_POS.y = y;
    MOUSE_POS.button = button;
    MOUSE_POS.handled = false;
    MOUSE_POS.pressTime = base.os.clock();
    if 1 == button then
        handleLeftMouseDown(x,y);
    elseif 2 == button then
        startRuler(x, y)
    elseif 4 == button then
        ZoomIn(x, y);
    elseif 5 == button then
        ZoomOut(x, y)
    end
    MapState.onMouseDown(self, x, y, button)
    --base.debug.sethook ();
end

-------------------------------------------------------------------------------
function panState_onMouseDrag(self, dx, dy, button, x, y)
    MOUSE_POS = MOUSE_POS or { pressTime = 0; };
    MOUSE_POS.drag = true;    
    --base.print('dx, dy, button, x, y',dx, dy, button, x, y);
    --base.U.traverseTable(MOUSE_POS,1);
    if ( (base.os.clock() - MOUSE_POS.pressTime) < 0.2 ) and ( button ~= 3 ) then
        --base.print('skipping !!!');
        return;
    end;
    if button == 2 then
      moveRuler(dx, dy, x, y)
    elseif button == 1 then
      -- Выбранные точки маршрута можно двигать, но делать это нужно в специальном
      -- состоянии редактирования точки маршрута, когда открыта панель ROUTE и нажата кнопка EDIT.
      --handleLeftMouseDown(MOUSE_POS.x, MOUSE_POS.y);
      if base.panel_route.window:isVisible() and base.panel_route.b_edit:getState() then
        waypointDragged = base.panel_route.vdata.wpt
        if selectedGroup then
          local lat, long = mapView:getMapPoint(x, y)
          -- Здесь тоже нужна проверка суша/море.

          -- Поскольку не везде объект можно поставить, то в onMouseUp() нужно проверять
          -- тип поверхности и отменять операцию перемещения, если тип не подходящий.
          local scale = mapView:getScale()
          if base.module_mission.UNITS_SCALE < scale then
                if MOUSE_POS.lastPickedObject
                    and MOUSE_POS.lastPickedObject.subclass == 'unit' 
                    and (selectedUnit ~= nil) then
                    move_unit(selectedGroup, selectedUnit, lat, long);
                else
                    local wptIndex = base.panel_route.vdata.wpt.index;
                    move_waypoint(selectedGroup, wptIndex, lat, long, nil, true);
                end;
          else
            if selectedUnit then
              move_unit(selectedGroup, selectedUnit, lat, long)
              --waypointDragged = nil
            else
              local wptIndex = base.panel_route.vdata.wpt.index;
              move_waypoint(selectedGroup, wptIndex, lat, long, nil, true);
            end;
          end
          return
        end
      elseif ((base.panel_static.window:isVisible())
			and (not base.isPlannerMission())) then
        if selectedGroup then
          local lat, long = mapView:getMapPoint(x, y)
          -- Здесь тоже нужна проверка суша/море.
          move_group(selectedGroup, lat, long)
          return
        end
      elseif ((base.panel_targeting.window:isVisible())
			and (not base.isPlannerMission())) then
        if selectedGroup then
          local lat, long = mapView:getMapPoint(x, y)
          move_target(base.panel_targeting.vdata.target, lat, long)
        end
      elseif ((base.panel_zone.window:isVisible())
			and (not base.isPlannerMission())) then
        if base.panel_zone.currentZone then
          local lat, long = mapView:getMapPoint(x, y)
          move_zone(base.panel_zone.currentZone, lat, long)
        end
      elseif base.panel_fix_points.window:isVisible() then
        local lat, long = mapView:getMapPoint(x, y)
        if selectedGroup and selectedGroup.INUFixPoints then 
            for i,v in base.pairs(selectedGroup.INUFixPoints) do
                local point = selectedGroup.INUFixPoints[i];
                if point ==  base.panel_fix_points.vdata.selectedPoint then
                    move_INU_Fix_Point(selectedGroup, i, lat, long);
                    break;
                end;
            end;            
        end;
      end
    end
    MapState.onMouseDrag(self, dx, dy, button, x, y)
end

-------------------------------------------------------------------------------
function panState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
function setPanStateCallbacks()
  -- настраиваем реакцию карты на события мыши
  panState.onMouseUp = panState_onMouseUp;
  panState.onMouseDown = panState_onMouseDown;
  panState.onMouseDrag = panState_onMouseDrag;
  panState.onMouseMove = panState_onMouseMove;
end

-------------------------------------------------------------------------------
function creatingPlaneState_onMouseDown(self, x, y, button)
	if button == 1 then
		createAircraft('plane', x, y)
		return;
	end
	panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function creatingPlaneState_onMouseDrag(self, dx, dy, button, x, y)
    panState:onMouseDrag(dx, dy, button, x, y)
end

-------------------------------------------------------------------------------
function creatingPlaneState_onMouseUp(self, x, y, button)
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function creatingPlaneState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
  -- Функции создания самолетной группы
function setCreatingPlaneStateCallbacks()
  -- Данная функция создает новую самолетную группу миссии в указанном на карте месте,
  -- создает соответствующие ей объекты карты, передает данные о группе в
  -- соответствующие диалоги планирования группы и открывает эти диалоги.
  creatingPlaneState.onMouseDown = creatingPlaneState_onMouseDown;

  creatingPlaneState.onMouseDrag = creatingPlaneState_onMouseDrag;

  creatingPlaneState.onMouseUp = creatingPlaneState_onMouseUp;

  creatingPlaneState.onMouseMove = creatingPlaneState_onMouseMove;
end

-------------------------------------------------------------------------------
function creatingHelicopterState_onMouseDown(self, x, y, button)
	if button == 1 then
		createAircraft('helicopter', x, y)
		return;
	end
	panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function creatingHelicopterState_onMouseDrag(self, dx, dy, x, y, button)
	panState:onMouseDrag(dx, dy, x, y, button)
end

-------------------------------------------------------------------------------
function creatingHelicopterState_onMouseUp(self, x, y, button)
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function creatingHelicopterState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
-- Функции создания вертолетной группы
function setCreatingHelicopterStateCallbacks()
  -- Данная функция создает новую вертолетную группу миссии в указанном на карте месте,
  -- создает соответствующие ей объекты карты, передает данные о группе в
  -- соответствующие диалоги планирования группы и открывает эти диалоги.
    creatingHelicopterState.onMouseDown = creatingHelicopterState_onMouseDown
    creatingHelicopterState.onMouseDrag = creatingHelicopterState_onMouseDrag
    creatingHelicopterState.onMouseUp = creatingHelicopterState_onMouseUp
    creatingHelicopterState.onMouseMove = creatingHelicopterState_onMouseMove
end

-------------------------------------------------------------------------------
function creatingShipState_onMouseDown(self, x, y, button)
    if button == 1 then
      unselectAll()
      -- Создаем новый корабль на основании данных,
      -- заданных в соответствующей панели.
      local lat, long = mapView:getMapPoint(x, y)
      if not isInMap(lat, long) then
          showWarningWindow(cdata.placeInside)
          return
      end
      
      if 'sea' ~= getSurfaceType(lat, long, 0.00001) then
        showWarningWindow(_('Place ships in the sea, please'))
        
        return
      end
      
      base.toolbar.form.b_ship:setState(false)
      base.panel_ship.setSafeMode(false)
      base.panel_route.setSafeMode(false)
      base.panel_route.enableAlt(false)
      local vd = base.panel_ship.vdata
      local rvd = base.panel_route.vdata
      local tvd = base.panel_targeting.vdata
      local svd = base.panel_summary.vdata
	  vd.task = base.panel_ship.cdata.new_group_task;    -- Нужна умалчиваемая задача для данного типа группы
	  
      -- При добавлении точек группа должна быть известна как здесь, так и в диалогах группы.
      local group = base.module_mission.create_group(vd.country, 'ship', vd.name, base.panel_briefing.vdata.start_time, lat, long)
      group.task = vd.task
	  
      selectedGroup = group
      vd.group = group
      vd.name = group.name
      vd.unit.cur = 1
      vd.unit.number = 1
      vd.skills = { vd.skills[1] }
      vd.types = { vd.types[1] }
      rvd.group = group
      tvd.group = group
      tvd.target = nil
      svd.group = group
      local unit = base.module_mission.insert_unit(group, vd.types[1], 
            vd.skills[1], 1, vd.names[1])
      local name = unit.name
      rvd.wpt = base.module_mission.insert_waypoint(group, 1, 
          base.panel_route.actions.turningPoint, lat, long, 0, 20/3.6)
      rvd.wpt.eta = base.panel_briefing.vdata.start_time
      set_group_color(group, group.boss.boss.selectGroupColor)
      set_unit_color(unit, group.boss.boss.selectUnitColor)
      set_waypoint_color(rvd.wpt, group.boss.boss.selectWaypointColor)
      set_route_line_color(group, group.boss.boss.selectGroupColor)
      base.module_mission.update_group_map_objects(group)
      mapView:updateUserList(true)

      base.panel_ship.vdata.group = group

      mapView:setState(addingWaypointState)
      base.statusbar.updateState()
      base.panel_route.b_add:setState(true)
      base.panel_route.b_edit:setState(false)
      base.panel_ship.update()
      base.panel_targeting.update(false)
      base.panel_route.update()
      return
    end
    panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function creatingShipState_onMouseDrag(self, dx, dy, x, y, button)
    panState:onMouseDrag(dx, dy, x, y, button)
end

-------------------------------------------------------------------------------
function creatingShipState_onMouseUp(self, x, y, button)
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function creatingShipState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
  -- Функции создания корабельной группы
function setCreatingShipStateCallbacks()
  -- Данная функция создает новую корабельную группу миссии в указанном на карте месте,
  -- создает соответствующие ей объекты карты, передает данные о группе в
  -- соответствующие диалоги планирования группы и открывает эти диалоги.
  creatingShipState.onMouseDown = creatingShipState_onMouseDown;
  creatingShipState.onMouseDrag = creatingShipState_onMouseDrag;
  creatingShipState.onMouseUp = creatingShipState_onMouseUp;
  creatingShipState.onMouseMove = creatingShipState_onMouseMove;
end

-------------------------------------------------------------------------------
function creatingVehicleState_onMouseDown(self, x, y, button)
    if button == 1 then
      unselectAll()
      base.toolbar.form.b_vehicle:setState(false)
      -- Создаем новую колонну машинок на основании данных,
      -- заданных в соответствующей панели.
      local lat, long = mapView:getMapPoint(x, y)
      if not isInMap(lat, long) then
          showWarningWindow(cdata.placeInside)
          return
      end
      
      if 'land' ~= getSurfaceType(lat, long, 0.00001) then
        showWarningWindow(_('Place ground vehicles on the land, please'))
        
        return
      end 
      
      base.panel_vehicle.setSafeMode(false)
      base.panel_route.setSafeMode(false)
      local vd = base.panel_vehicle.vdata
      vd.task = base.panel_vehicle.cdata.new_group_task;    -- Нужна умалчиваемая задача для данного типа группы

      local rvd = base.panel_route.vdata
      local tvd = base.panel_targeting.vdata
      local svd = base.panel_summary.vdata

      -- При добавлении точек группа должна быть известна
      local group = base.module_mission.create_group(vd.country, 'vehicle', vd.name, base.panel_briefing.vdata.start_time, lat, long)
      group.task = vd.task
      selectedGroup = group
      vd.group = group
      vd.name = group.name
	  vd.task = group.task
      vd.unit.cur = 1
      vd.unit.number = 1
      vd.skills = { vd.skills[1] }
      vd.types = { vd.types[1] }
      rvd.group = group
      tvd.group = group
      tvd.target = nil
      svd.group = group

      local unit = base.module_mission.insert_unit(group, vd.types[1], 
            vd.skills[1], 1, vd.names[1])
      local name = unit.name
      local alt = U.getAltitude(lat, long)
      rvd.wpt = base.module_mission.insert_waypoint(group, 1, 
                base.panel_route.actions.offRoad, lat, long, alt, 20/3.6)
      rvd.wpt.eta = base.panel_briefing.vdata.start_time
      set_group_color(group, group.boss.boss.selectGroupColor)
      set_unit_color(unit, group.boss.boss.selectUnitColor)
      set_waypoint_color(rvd.wpt, group.boss.boss.selectWaypointColor)
      set_route_line_color(group, group.boss.boss.selectGroupColor)

      base.module_mission.update_group_map_objects(group)
      mapView:updateUserList(true)

      base.panel_vehicle.vdata.group = group

      mapView:setState(addingWaypointState)
      base.statusbar.updateState()
      base.panel_route.enableAlt(false)
      base.panel_route.b_add:setState(true)
      base.panel_route.b_edit:setState(false)

      base.panel_vehicle.update()
      base.panel_targeting.update(false)
      base.panel_route.update()
      return
    end
    panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function creatingVehicleState_onMouseDrag(self, dx, dy, x, y, button)
    panState:onMouseDrag(dx, dy, x, y, button)
end

-------------------------------------------------------------------------------
function creatingVehicleState_onMouseUp(self, x, y, button)
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function creatingVehicleState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
-- Функции создания наземной группы
function setCreatingVehicleStateCallbacks()
  -- Данная функция создает новую наземную группу миссии в указанном на карте месте,
  -- создает соответствующие ей объекты карты, передает данные о группе в
  -- соответствующие диалоги планирования группы и открывает эти диалоги.
  creatingVehicleState.onMouseDown = creatingVehicleState_onMouseDown
  creatingVehicleState.onMouseDrag = creatingVehicleState_onMouseDrag
  creatingVehicleState.onMouseUp = creatingVehicleState_onMouseUp;
  creatingVehicleState.onMouseMove = creatingVehicleState_onMouseMove;
end

-------------------------------------------------------------------------------
function addingWaypointState_onMouseDown(self, x, y, button)
    if button == 1 then
      -- Добавляем новую точку маршрута после текущей точки в миссию.
      -- Панель маршрута для текущей группы предполагается открытой.
      local group = base.panel_route.vdata.group
      selectedGroup = group
      local lat, long = mapView:getMapPoint(x, y)
      if not isInMap(lat, long) then
          showWarningWindow(cdata.placeInside)
          return
      end
      local surfaceType = getSurfaceType(lat, long, 0.00001)
      
      if 'ship' == group.type and 'sea' ~= surfaceType then
        showWarningWindow(_('Place ships in the sea, please'))
        
        return
      elseif 'vehicle' == group.type and 'land' ~= surfaceType then
        showWarningWindow(_('Place ground vehicles on the land, please'))
        
        return        
      end

      local rvd = base.panel_route.vdata
      local ind = rvd.wpt.index + 1
      local type = base.panel_route.actions.turningPoint
      if group.type == 'vehicle' then
          type = base.panel_route.actions.offRoad
          if 'On Road' == rvd.wpt.type.action then
              type = base.panel_route.actions.onRoad
          end
		  if 'Rank' == rvd.wpt.type.action then
              type = base.panel_route.actions.rank
          end
		  if 'Cone' == rvd.wpt.type.action then
              type = base.panel_route.actions.cone
          end
		  
      end
      local speed = rvd.wpt.speed
      local height = U.getAltitude(lat, long)
      local alt = rvd.wpt.alt
      if group.type == 'plane' then
        alt = base.math.max(height, alt)
      elseif group.type == 'helicopter' then
        alt = base.math.max(height, alt)
      elseif group.type == 'ship' then
        alt = height
      elseif group.type == 'vehicle' then
        alt = height
      end
      rvd.wpt = base.module_mission.insert_waypoint(group, ind, type, lat, long, alt, speed)
      rvd.wpt.eta = base.panel_briefing.vdata.start_time
      base.panel_route.update()
      base.panel_summary.update()
      -- Обновляем объекты карты.
      set_waypoints_color(group, group.boss.boss.selectGroupColor)
      set_waypoint_color(rvd.wpt, group.boss.boss.selectWaypointColor)
      base.module_mission.update_group_map_objects(group)
      mapView:updateUserList(true)
      return
    end
    panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function addingWaypointState_onMouseDrag(self, dx, dy, button, x, y)
	panState:onMouseDrag(dx, dy, button, x, y)
end

-------------------------------------------------------------------------------
function addingWaypointState_onMouseUp(self, x, y, button)
    local rvd = base.panel_route.vdata;
    --base.print(rvd, rvd.wpt, rvd.wpt.action);
    if rvd and rvd.wpt and ('On Road' == rvd.wpt.type.action) then
        base.module_mission.move_waypoint_to_road(rvd.wpt);
    end;
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function addingWaypointState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
  -- Функции добавления точек маршрута
function setAddingWaypontStateCallbacks()
  -- Данная функция добавляет новую точку маршрута после текущей точки,
  -- создает соответствующие объекты карты,
  -- передает данные в миссию и в диалог маршрута.
  -- Самая первая точка создается при создании группы на карте.

  addingWaypointState.onMouseDown = addingWaypointState_onMouseDown;
  addingWaypointState.onMouseDrag = addingWaypointState_onMouseDrag
  addingWaypointState.onMouseMove = addingWaypointState_onMouseMove
  addingWaypointState.onMouseUp = addingWaypointState_onMouseUp
end

-------------------------------------------------------------------------------
function addingTargetState_onMouseDown(self, x, y, button)
    if button == 1 then
      local lat, long = mapView:getMapPoint(x, y)
      local tvd = base.panel_targeting.vdata
      local group = tvd.group
--      base.print(group.name)
      selectedGroup = group
      local rvd = base.panel_route.vdata
      local wpt = rvd.wpt
      if not wpt.targets then
        wpt.targets = {}
      end
      local targets = wpt.targets
      local ind = #targets+1
      tvd.target = base.module_mission.insert_target(wpt, ind, lat, long, tvd.radius)
      --set_targets_color(wpt, group.boss.boss.selectGroupColor)
      base.panel_targeting.selectTargetByIndex(0); -- сброс выделения
      set_target_color(tvd.target, group.boss.boss.selectWaypointColor)
      base.module_mission.update_group_map_objects(group)
      mapView:updateUserList(true)
	  base.panel_targeting.setDefaultCategories()
      base.panel_targeting.update()
      return
    end
    panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function addingTargetState_onMouseDrag(self, dx, dy, button, x, y)
    panState:onMouseDrag(dx, dy, button, x, y)
end

-------------------------------------------------------------------------------
function addingTargetState_onMouseUp(self, x, y, button)
	panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function addingTargetState_onMouseMove(self, x, y)
	moveCursor(x, y)
end

-------------------------------------------------------------------------------
-- Функции создания целей на маршруте
function setAddingTargetStateCallbacks()
  -- Данная функция создает новую цель текущей точки маршрута в указанном на карте месте,
  -- создает соответствующие ей объекты карты, передает данные о цели в
  -- соответствующий диалог планирования целей.

  addingTargetState.onMouseDown = addingTargetState_onMouseDown
  addingTargetState.onMouseDrag = addingTargetState_onMouseDrag
  addingTargetState.onMouseUp = addingTargetState_onMouseUp
  addingTargetState.onMouseMove = addingTargetState_onMouseMove
end

-------------------------------------------------------------------------------
function creatingZoneState_onMouseDown(self, x, y, button)
    if button == 1 then
      base.panel_zone.setSafeMode(false)
      base.toolbar.form.b_zone:setState(false)
      local lat, long = mapView:getMapPoint(x, y)
      if not isInMap(lat, long) then
          showWarningWindow(cdata.placeInside)
          return
      end
      local panel = base.panel_zone
      if not base.module_mission.mission.triggers.zones then
        base.module_mission.mission.triggers.zones = {}
      end
      local index = #base.module_mission.mission.triggers.zones+1
      local zone = base.module_mission.create_trigger_zone(_('New Trigger Zone'), lat, long, panel.defaultRadius, panel.color)
      base.table.insert(base.module_mission.mission.triggers.zones, zone)
      panel.currentZone = zone
      panel.show(true)
      setPanState()
      return
    end
    panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function creatingZoneState_onMouseDrag(self, dx, dy, button, x, y)
    panState:onMouseDrag(dx, dy, button, x, y)
end

-------------------------------------------------------------------------------
function creatingZoneState_onMouseUp(self, x, y, button)
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function creatingZoneState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
-- Функции создания зон для триггеров
function setCreatingZoneStateCallbacks()
  creatingZoneState.onMouseDown = creatingZoneState_onMouseDown

  creatingZoneState.onMouseDrag = creatingZoneState_onMouseDrag
  creatingZoneState.onMouseUp = creatingZoneState_onMouseUp
  creatingZoneState.onMouseMove = creatingZoneState_onMouseMove
end

-------------------------------------------------------------------------------
function creatingTemplateState_onMouseDown(self, x, y, button)
    if button == 1 then

      unselectAll()

      local lat, long = mapView:getMapPoint(x, y)
      if not isInMap(lat, long) then
          showWarningWindow(cdata.placeInside)
          return
      end

      local _panel = base.panel_template

      local _country_name = _panel.c_country:getText()
      if not _country_name or (#_country_name < 1) then return end

      local c = db.country_by_name[_country_name]
      if not c then return end
      
      local _template = _panel.templates[c.WorldID]
      if not _template then return end

      local group_name = 'New Group'
      
      local type = _template.type[_panel.c_template:getText()]
	  
      
      if not type or (#type < 1) then return end
      if type == 'vehicle' then
        group_name = base.panel_vehicle.vdata.name
        
        if 'land' ~= getSurfaceType(lat, long, 0.00001) then
          showWarningWindow(_('Place ground objects on the land, please'))
          return
        end
      
      elseif type == 'ship' then
        group_name = base.panel_ship.vdata.name
        
        if 'sea' ~= getSurfaceType(lat, long, 0.00001) then
          showWarningWindow(_('Place ships in the sea, please'))
          return
        end

      elseif (type == 'helicopter') or (type == 'plane') then
        base.panel_aircraft.setView(type);
        group_name = base.panel_aircraft.vdata.name
      else
        group_name = _('New Static Object')
      end

      local _template_name = _panel.c_template:getText()
      if not _template_name or (#_template_name < 1) then return end

      local _group = _template.groups[_template_name]
      if not _group or #_group < 1 then return end	  

      local group = base.module_mission.create_group(_panel.c_country:getText(), _template.type[_panel.c_template:getText()], group_name, base.panel_briefing.vdata.start_time, lat, long)
      if not group then return end	  	  	 
	  
      -- saving template name, just in case
      group.template_name = _template_name
      selectedGroup = group
    
      local alt = U.getAltitude(lat, long)

      local i = 1
	  local j = 1
      while _group[i] do

        local v = _group[i]

        v.skill = v.skill or "Good"
        v.heading = v.heading or 0			

        local _heading = _panel.d_heading:getValue() * base.math.pi / 180 
        local x_,y_ = base.MapWindow.mapView:convertRadiansToMeters(lat, long)
        local dx_,dy_ = base.MapWindow.mapView:convertRadiansToMeters(lat+v.lat, long+v.long)
        local _dlat = dx_-x_
        local _dlong = dy_-y_
        local _new_x = x_ + _dlat*base.math.cos(_heading) - _dlong*base.math.sin(_heading)
        local _new_y = y_ + _dlat*base.math.sin(_heading) + _dlong*base.math.cos(_heading)
        local mapX, mapY = base.MapWindow.mapView:convertMetersToRadians(_new_x, _new_y)
        local wrong_position = false
        
        if type == 'vehicle' then
            if 'land' ~= getSurfaceType(mapX, mapY, 0.00001) then wrong_position = true end
        elseif type == 'ship' then
            if 'sea' ~= getSurfaceType(mapX, mapY, 0.00001) then wrong_position = true end
        end

        if not wrong_position then 
            local unit = base.module_mission.insert_unit(group, db.displayNameByName[v.name], crutches.idToSkill(v.skill), j, group_name, mapX, mapY, base.math.mod(_heading + v.heading,2*base.math.pi))
            if type == 'static' then
                unit.shape_name = base.me_db.unit_by_name[db.displayNameByName[v.name]].ShapeName
                unit.category = v.category
            end
		    j = j +1
        end
		
		if (type == 'helicopter') or (type == 'plane') then
			group.units[i].payload 		= _group[i].payload
			group.units[i].livery_id 	= _group[i].livery_id
		end
		
        i = i +1
      end

		--base.print('i=',i)
		--base.U.traverseTable(group.units)
		local wpt 
		local alt_air = _group[1].alt
		if (alt_air < (alt + 200)) then
			alt_air = alt + 200
		end
		
		group.task = _group[1].task
		
		if (type == 'helicopter') or (type == 'plane') then
			wpt = base.module_mission.insert_waypoint(group, 1, _group[1].actions, lat, long, alt_air, _group[1].speed)
		else
			wpt = base.module_mission.insert_waypoint(group, 1, _group[1].actions, lat, long, alt, _group[1].speed)
		end
		
		base.module_mission.update_group_map_objects(group)
		mapView:updateUserList(true)

		return
    end
    panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function creatingTemplateState_onMouseDrag(self, dx, dy, button, x, y)
    panState:onMouseDrag(dx, dy, button, x, y)
end

-------------------------------------------------------------------------------
function creatingTemplateState_onMouseUp(self, x, y, button)
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function creatingTemplateState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
function setCreatingTemplateStateCallbacks()
  creatingTemplateState.onMouseDown = creatingTemplateState_onMouseDown;
  creatingTemplateState.onMouseDrag = creatingTemplateState_onMouseDrag
  creatingTemplateState.onMouseUp = creatingTemplateState_onMouseUp
  creatingTemplateState.onMouseMove = creatingTemplateState_onMouseMove
end

-------------------------------------------------------------------------------
function tapeState_onMouseDown(self, x, y, button)
    if button == 1 then
        startRuler(x, y)
        return
    end
    panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function tapeState_onMouseUp(self, x, y, button)
    if button == 1 then
      moveRuler(0, 0, x, y)
      return
    end
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function tapeState_onMouseDrag(self, dx, dy, button, x, y)
    if button == 1 then
        moveRuler(dx, dy, x, y)
    end
    panState:onMouseDrag(dx, dy, button, x, y)
end

-------------------------------------------------------------------------------
function tapeState_onMouseMove(self, x, y)
    panState:onMouseMove(x, y)
end

-------------------------------------------------------------------------------
  -- Функции измерительной линейки
function setTapeStateCallbacks()
  tapeState.onMouseDown = tapeState_onMouseDown;
  tapeState.onMouseUp = tapeState_onMouseUp;
  tapeState.onMouseDrag = tapeState_onMouseDrag;
  tapeState.onMouseMove = tapeState_onMouseMove;
  -- function tapeState:onMouseClick(x, y, button)
    -- panState:onMouseClick(x, y, button)
  -- end
end

-------------------------------------------------------------------------------
function removeTapeObjects()
  if tape then
    mapView:removeUserObjects({tape, tape_text})
    mapView:updateUserList(true)
    Mission.mapObjects[tape.id] = nil
    Mission.mapObjects[tape_text.id] = nil
    tape = nil
    tape_text = nil
  end
end

-------------------------------------------------------------------------------
function move_unit(group, unit, lat, long, doNotRedraw)
  -- Нужно пересчитать координаты у самой точки, у точки линии маршрута, у номера точки,
  -- а также у юнитов, если точка маршрута - первая.
  --base.print('\nmove_unit ====================:');
  if unit.index == 1 then 
    base.U.stack('Achtung!!! index == 1 move_unit');
    base.assert(0);
  end;
  --base.print('doNotRedraw',doNotRedraw);
  --base.U.traverseTable(unit, 2);
  if not checkSurface(group, lat, long) then 
    return
  end;  
  unit.lat = lat
  unit.long = long
  local units = group.mapObjects.units
  --base.print('unit.index', unit.index, units)
  --base.U.traverseTable(units[unit.index], 5);
  units[unit.index].points[1][1]["x"] = lat
  units[unit.index].points[1][1]["y"] = long
  
  if not doNotRedraw then
        local objects = {}
        base.table.insert(objects, units[unit.index])
        base.module_mission.updateUnitZones(unit);
        -- for i,unit in base.ipairs(group.units) do 
            -- base.module_mission.updateUnitZones(unit);
        -- end
        --base.print('unit.linkChildren', unit.linkChildren);
      if unit.linkChildren then
        for i,wpt in base.ipairs(unit.linkChildren) do 
            move_waypoint(wpt.boss, wpt.index, lat, long, nil, true);
        end
      end;
      
      mapView:removeUserObjects(objects)
      mapView:addUserObjects(objects)
      mapView:updateUserList(true)
  end
end

-------------------------------------------------------------------------------
function move_INU_Fix_Point(group, index, lat, long)
  -- Нужно пересчитать координаты у самой точки, у точки линии маршрута, у номера точки,
  -- а также у юнитов, если точка маршрута - первая.
  local objects = {}
  local pt = group.INUFixPoints[index]
  pt.lat = lat
  pt.long = long

  local points = {
            {
                {x = pt.lat, y = pt.long},
            },
        }
  local symbol = group.mapObjects.INUFixPoints[index];
  symbol.points = points;
  base.table.insert(objects, symbol)

  local coeff = mapView:getScale()/100000  
  local text = group.mapObjects.INUFixPoints_numbers[index];
  points = {
            {
                {x = pt.lat+0.00005*coeff, y = pt.long+0.00005*coeff},
                {x = pt.lat+0.00005*coeff, y = pt.long+0.00105*coeff},
            },
        }
  text.points = points;
  base.table.insert(objects, text)
  
  mapView:removeUserObjects(objects)
  mapView:addUserObjects(objects)
  mapView:updateUserList(true)

  base.panel_fix_points.update()
end

-------------------------------------------------------------------------------
function isInMap(lat, long)
    return nil ~= MapView:convertRadiansToMeters(lat, long) 
end

-------------------------------------------------------------------------------
function checkSurface(group, lat, long, showWarning, callback)
    local displayMessage = callback or showWarningWindow;
    if not isInMap(lat, long) then
        return false
    end

    local surfaceType = getSurfaceType(lat, long, 0.00001)
    local type = '';
    
    if 'ship' == group.type then
        type = 'sea_object';
    elseif 'vehicle' == group.type then    
        type = 'land_object';
    elseif 'static' == group.type then    
        local unit = group.units[1];
        if _('Ships') == unit.category then
            type = 'sea_object';
        else
            local unit =  db.unit_by_name[unit.type]
            if (unit.SeaObject ~= nil) and (unit.SeaObject == true) then
                type = 'sea_object';
            else
                type = 'land_object';
            end;
        end;
    end;

    local result = true
    
    if 'sea_object' == type then
      if 'sea' ~= surfaceType then
        if showWarning then
          displayMessage(_('Place ships in the sea, please'))
        end
        
        result = false
      end
    elseif 'land_object' == type then
      if 'land' ~= surfaceType then
        if showWarning then
          displayMessage(_('Place ground objects on the land, please'))
        end
        
        result = false
      end
    end  
          
    return result;
end;

-------------------------------------------------------------------------------
function move_waypoint(group, index, lat, long, dontMoveLinked, doNotUpdateRoute)
  -- Нужно пересчитать координаты у самой точки, у точки линии маршрута, у номера точки,
  -- а также у юнитов, если точка маршрута - первая.
  -- base.print('move_waypoint group, index, lat, long, dontMoveLinked, doNotUpdateRoute', 
     -- group, index, lat, long, dontMoveLinked, doNotUpdateRoute);
  --base.U.stack('move_waypoint');
  local objects = {}
  if not group.route then
      return
  end
  local wpt = group.route.points[index]
  if not wpt then
      return
  end

  if not checkSurface(group, lat, long) then 
    return
  end;
  if index == 1 then
      for i = 2, #group.units do
        local unit = group.units[i];
        local dLat = lat - group.lat
        local dLong = long - group.long  
        if not checkSurface(group, unit.lat + dLat, unit.long + dLong) then 
            return
        end;
      end;
  end;
  
  wpt.lat = lat
  wpt.long = long
  
  -- Нужно скорректировать перегоны
  if group.route.spans and #group.route.spans > 0 then
    local spans = group.route.spans
    -- Если точка не первая, то модифицируется перегон с предыдущим индексом, который располагается перед данной точкой.
    if index > 1 then
      local p = group.route.points[index-1]
      spans[index-1] = {{lat=p.lat,long=p.long},{lat=lat,long=long}}
    end
    -- Если точка не последняя, то модифицируется перегон с текущим индексом, который располагается после данной точки.
    if index < #group.route.points then
      local p = group.route.points[index+1]
      spans[index] = {{lat=lat,long=long},{lat=p.lat,long=p.long}}
    end
  end
  base.module_mission.build_route_line(group)
  local scale = mapView:getScale()
  local coeff = scale/100000
  local route = group.mapObjects.route
  base.table.insert(objects, route.line)
  route.points[index].points[1][1].x = lat
  route.points[index].points[1][1].y = long
  base.table.insert(objects, route.points[index])
  route.numbers[index].points[1][1].x = lat+0.00005*coeff
  route.numbers[index].points[1][1].y = long+0.00005*coeff
  route.numbers[index].points[1][2].x = lat+0.00005*coeff
  route.numbers[index].points[1][2].y = long+0.00105*coeff
  base.table.insert(objects, route.numbers[index])
  if route.targetLines[index] then
    for i=1,#route.targetLines[index] do
      route.targetLines[index][i].points[1][1].x = lat
      route.targetLines[index][i].points[1][1].y = long
      base.table.insert(objects, route.targetLines[index][i])
    end
  end
  if index == 1 then
    local dLat = lat - group.lat
    local dLong = long - group.long
    group.lat = lat
    group.long = long

        local units = group.mapObjects.units
        for i=1, #units do
            local unit = group.units[i]
            local udb = base.me_db.unit_by_CLSID[unit.CLSID]
            local hasZone = false;
            if udb then
                if (udb.DetectionRange and (udb.DetectionRange > 0)) 
                    or (udb.ThreatRange and (udb.ThreatRange > 0)) then
                    hasZone = true;
                end
            end
            if unit.index ~= 1 then
                move_unit(group, unit, unit.lat + dLat, unit.long + dLong, false);
                if ((base.module_mission.UNITS_SCALE >= scale) and (1 ~= i)) or hasZone then
                    base.table.insert(objects, units[i])
                end
            else -- для первого юнита ничего не рисуем, вместо юнита рисуется первая точка маршрута со значком юнита
                unit.lat = lat
                unit.long = long
                group.mapObjects.units[1].points[1][1]["x"] = lat
                group.mapObjects.units[1].points[1][1]["y"] = long     
                if unit.linkChildren then
                    for i,wpt in base.ipairs(unit.linkChildren) do 
                        move_waypoint(wpt.boss, wpt.index, lat, long, nil, true);
                    end
                end;

            end;
            base.module_mission.updateUnitZones(unit);
        end
  end

  -- if group.units[1].linkChildren and (1 == index) and (not dontMoveLinked) then
      -- for _tmp, _waypoint in base.pairs(group.units[1].linkChildren) do
          -- if _waypoint.linkUnit then
            -- base.print('_waypoint.linkUnit');
            -- base.U.traverseTable(_waypoint.linkUnit, 1);
            -- move_waypoint(_waypoint.boss, _waypoint.index, _waypoint.linkUnit.lat, _waypoint.linkUnit.long, true, doNotUpdateRoute)
          -- else
            -- move_waypoint(_waypoint.boss, _waypoint.index, wpt.lat, wpt.long, true, doNotUpdateRoute)
          -- end;
      -- end
  -- end

  mapView:removeUserObjects(objects)
  mapView:addUserObjects(objects)
  mapView:updateUserList(true)

  if not doNotUpdateRoute then
      base.module_mission.calc_route_length(group)
      base.panel_summary.update()
      base.panel_route.update()
  end
end

-------------------------------------------------------------------------------
function move_target(target, lat, long)
  local scale = mapView:getScale()
  local coeff = scale/100000
  -- Изменяются координаты цели.
  target.lat = lat
  target.long = long
  local wpt = target.boss
  local group = wpt.boss
  local objects = {}
  local trg = group.mapObjects.route.targets[wpt.index][target.index]
  -- Изменяются координаты точки цели на карте.
  local p = trg.points[1][1]
  p.x = lat
  p.y = long
  base.table.insert(objects, trg)
  -- Изменяются координаты номера цели на карте.
  local num = group.mapObjects.route.targetNumbers[wpt.index][target.index]
  local p1 = num.points[1][1]
  local p2 = num.points[1][2]
  p1.x = lat+0.00005*coeff
  p1.y = long+0.00005*coeff
  p2.x = lat+0.00005*coeff
  p2.y = long+0.00105*coeff
  base.table.insert(objects, num)
  -- Изменяются координаты конца линии цели на карте.
  local line = group.mapObjects.route.targetLines[wpt.index][target.index]
  p = line.points[1][2]
  p.x = lat
  p.y = long
  base.table.insert(objects, line)
  -- Пересчитываются координаты точек зоны цели на карте.
  local zone = group.mapObjects.route.targetZones[wpt.index][target.index]
  if zone then
    base.module_mission.update_target_zone(target)
    -- Зона обновляется на карте внутри функции.
  end
  -- Обновляются соответствующие объекты карты.
  mapView:removeUserObjects(objects)
  mapView:addUserObjects(objects)
  mapView:updateUserList(true)
end

-------------------------------------------------------------------------------
function move_zone(zone, lat, long)
  local scale = mapView:getScale()
  local coeff = scale/100000
  -- Изменяются координаты цели.
  zone.lat = lat
  zone.long = long
  objects = {}
  -- Изменяются координаты цетра зоны на карте.
  local p = zone.mapObjects.center.points[1][1]
  p.x = lat
  p.y = long
  base.table.insert(objects, zone.mapObjects.center)
  -- Изменяются координаты названия зоны на карте.
  local name = zone.mapObjects.name
  local p1 = name.points[1][1]
  local p2 = name.points[1][2]
  p1.x = lat+0.00005*coeff
  p1.y = long+0.0001*coeff
  p2.x = lat+0.00005*coeff
  p2.y = long+0.0011*coeff
  base.table.insert(objects, name)
  -- Обновляются соответствующие объекты карты.
  mapView:removeUserObjects(objects)
  mapView:addUserObjects(objects)
  mapView:updateUserList(true)
  -- Пересчитываются координаты точек зоны на карте.
  base.module_mission.update_trigger_zone(zone)
end

-------------------------------------------------------------------------------
-- returns true if sufrace is good for FARP
function isValidFarpSurface(lat, long)
    return 20 >= U.getAltitudeDelta(lat, long, 261)
end

-------------------------------------------------------------------------------
-- Перемещает статическую группу
function move_group(group, lat, long)
  --base.U.stack('move_group');
  if not checkSurface(group, lat, long) then 
    return
  end;
  
  if ('static' == group.type) and (cdata.farp == group.units[1].type) then
      if not isValidFarpSurface(lat, long) then
          return
      end
  end

  group.lat = lat
  group.long = long
  group.units[1].lat = lat
  group.units[1].long = long
  group.mapObjects.units[1].points[1][1].x = lat
  group.mapObjects.units[1].points[1][1].y = long
  base.module_mission.update_group_map_objects(group)

  if group.units[1].linkChildren then
      for _tmp, wpt in base.pairs(group.units[1].linkChildren) do
          move_waypoint(wpt.boss, wpt.index, lat, long, true)
      end
  end
end

-------------------------------------------------------------------------------
function creatingStaticState_onMouseDown(self, x, y, button)
    if button == 1 then
      if selectedGroup then
        revert_selection(selectedGroup)
        selectedGroup = nil
      end
      base.toolbar.form.b_static:setState(false)
      -- Создаем новый статический объект на основании данных,
      -- заданных в соответствующей панели.
      local lat, long = mapView:getMapPoint(x, y)
      if not isInMap(lat, long) then
          showWarningWindow(cdata.placeInside)
          return
      end
      local vd = base.panel_static.vdata
      local surfaceType = getSurfaceType(lat, long, 0.00001)
      
      local unit =  db.unit_by_name[vd.type]
      if (vd.category == _('Ships')) or (unit.SeaObject == true)then -- юнит - корабль или водный статик
        if 'sea' ~= surfaceType then -- 
          if (vd.category == _('Ships')) then 
            showWarningWindow(_('Place ships in the sea, please'))          
            return
          else
            showWarningWindow(_('Place sea objects in the sea, please'))          
            return          
          end;
        end
      else -- юнит - НЕ корабль
        if ('land' ~= surfaceType)  then -- поверхность НЕ земля
            showWarningWindow(_('Place ground objects on the land, please'))          
            return            
        end
      end

      if cdata.farp == vd.type then
          if not isValidFarpSurface(lat, long) then
              local msgWindow = MsgWindow.new(
                        _('Too steep terrain'), 
                        _('Warning'), _('warning'), _('OK'))
              msgWindow:setVisible(true)
              return
          end
      end
      vd.lat = lat
      vd.long = long
      vd.alt = 0
      local group = base.module_mission.create_group(vd.country, 'static', vd.name, base.panel_briefing.vdata.start_time, vd.lat, vd.long)
      group.heading = vd.heading;
      vd.group = group;
      group.hidden = vd.hidden;
      selectedGroup = group
      local unit = base.module_mission.insert_unit(group, vd.type, vd.skill, 1, group.name)
      unit.heading = group.heading
      unit.shape_name = base.me_db.unit_by_name[vd.type].ShapeName
      unit.category = vd.category
      set_group_color(group, group.boss.boss.selectGroupColor)
      set_unit_color(group.units[1], group.boss.boss.selectUnitColor)
      base.module_mission.insert_waypoint(group, 1, '', lat, long, 0, 200/3.6)
      
      base.module_mission.update_group_map_objects(group)

      base.panel_static.update()
      mapView:setState(panState);
      return
    end
    panState:onMouseDown(x, y, button)
end

-------------------------------------------------------------------------------
function creatingStaticState_onMouseUp(self, x, y, button)
    panState:onMouseUp(x, y, button)
end

-------------------------------------------------------------------------------
function creatingStaticState_onMouseDrag(self, dx, dy, button, x, y)
    panState:onMouseDrag(dx, dy, button, x, y)
end

-------------------------------------------------------------------------------
function creatingStaticState_onMouseMove(self, x, y)
    moveCursor(x, y)
end

-------------------------------------------------------------------------------
function setCreatingStaticStateCallbacks()
  -- Данная функция создает новую статическую группу миссии в указанном на карте месте,
  -- создает соответствующие ей объекты карты, передает данные о группе в
  -- соответствующий диалог планирования группы и открывает этот диалог.

  creatingStaticState.onMouseDown = creatingStaticState_onMouseDown
  creatingStaticState.onMouseUp = creatingStaticState_onMouseUp
  creatingStaticState.onMouseDrag = creatingStaticState_onMouseDrag
  creatingStaticState.onMouseMove = creatingStaticState_onMouseMove
end

-------------------------------------------------------------------------------
function setCallbacks()
  setPanStateCallbacks()
  setCreatingPlaneStateCallbacks()
  setCreatingHelicopterStateCallbacks()
  setCreatingShipStateCallbacks()
  setCreatingVehicleStateCallbacks()
  setCreatingStaticStateCallbacks()
  setAddingWaypontStateCallbacks()
  setAddingTargetStateCallbacks()
  setCreatingZoneStateCallbacks()
  setCreatingTemplateStateCallbacks()
  setCreatingINUFixPointStateCallbacks()
  setTapeStateCallbacks()
  
  window:addKeyCombination(function()
        zoomIn()
        mapView:updateUserList(false)    
        end, 
    nil,'[+]');
  window:addKeyCombination(function()
        zoomOut()
        mapView:updateUserList(false)    
        end, 
    nil,'[-]');
  
end

-------------------------------------------------------------------------------
function creatingINUFixPointState_onMouseDown(self, x, y, button)
    if button == 1 then
         --unselectAll()
         local lat, long = mapView:getMapPoint(x, y);
          if not isInMap(lat, long) then
              showWarningWindow(cdata.placeInside)
              return
          end
         local group;
         if base.panel_aircraft.window:isVisible() and (base.panel_aircraft.getView == 'helicopter')  then
            group = base.panel_aircraft.vdata.group;            
         else
            return;
         end;
          local x, y = mapView:convertRadiansToMeters(lat, long);
          pt = base.module_mission.insert_INUFixPoint(group, lat, long, alt, group.boss.boss.selectGroupColor, x, y)
          base.module_mission.update_group_map_objects(group)
          mapView:updateUserList(true)
          base.panel_fix_points.vdata.selectedPoint = pt;
          base.panel_fix_points.update();
      return
    else
        panState:onMouseDown(x, y, button)
    end;
end;

-------------------------------------------------------------------------------
function creatingINUFixPointState_onMouseDrag(self, dx, dy, x, y, button)
    panState:onMouseDrag(dx, dy, x, y, button)
end;

-------------------------------------------------------------------------------
function creatingINUFixPointState_onMouseMove(self, x, y)
    moveCursor(x, y)
end;

-------------------------------------------------------------------------------
function creatingINUFixPointState_onMouseUp(self, x, y, button)
    --base.print('creatingINUFixPointState_onMouseUp')
    panState:onMouseUp(x, y, button)
end;

-------------------------------------------------------------------------------
function setCreatingINUFixPointStateCallbacks()
  creatingINUFixPointState.onMouseDown = creatingINUFixPointState_onMouseDown;
  creatingINUFixPointState.onMouseDrag = creatingINUFixPointState_onMouseDrag;
  creatingINUFixPointState.onMouseMove = creatingINUFixPointState_onMouseMove;
  creatingINUFixPointState.onMouseUp = creatingINUFixPointState_onMouseUp;

end;

-------------------------------------------------------------------------------
function zoomIn()
  local sind = mapView:getScaleIndex() - 1
  if 1 <= sind then
    local scale = scales[sind]
    setScale(scale)
    base.statusbar.updateScale()
    base.module_mission.scale_mission_map_objects(scale)
    if tape then
      local x = tape.points[1][2].x
      local y = tape.points[1][2].y
      local coeff = scale/100000
      tape_text.points[1][1] = {x=x+0.00005*coeff, y=y+0.0001*coeff}
      tape_text.points[1][2] = {x=x+0.00005*coeff, y=y+0.0011*coeff}
      mapView:removeUserObjects({tape_text})
      mapView:addUserObjects({tape_text})
      mapView:updateUserList(true)
    end
  end
end

-------------------------------------------------------------------------------
function zoomOut()
  local sind = mapView:getScaleIndex() + 1
  if #scales >= sind then
    local scale = scales[sind]
    setScale(scale)
    base.statusbar.updateScale()
    base.module_mission.scale_mission_map_objects(scale)
    if tape then
      local x = tape.points[1][2].x
      local y = tape.points[1][2].y
      local coeff = scale/100000
      tape_text.points[1][1] = {x=x+0.00005*coeff, y=y+0.0001*coeff}
      tape_text.points[1][2] = {x=x+0.00005*coeff, y=y+0.0011*coeff}
      mapView:removeUserObjects({tape_text})
      mapView:addUserObjects({tape_text})
      mapView:updateUserList(true)
    end
  end
end

-------------------------------------------------------------------------------
function moveCursor(x, y)
  local xr,yr = mapView:getMapPoint(x, y)
  base.statusbar.updateLatLong(xr, yr)
end

-------------------------------------------------------------------------------
function find_country(country_name)
  local coals = base.module_mission.mission.coalitions
  local coal
  for i,v in base.pairs(coals) do
    for j,u in base.pairs(v) do
      if u == country_name then
        if i == 'blue' then
          if base.module_mission.mission.coalition[1].name == 'BLUE' then
            coal = base.module_mission.mission.coalition[1]
          else
            coal = base.module_mission.mission.coalition[2]
          end
        else
          if base.module_mission.mission.coalition[1].name == 'BLUE' then
            coal = base.module_mission.mission.coalition[2]
          else
            coal = base.module_mission.mission.coalition[1]
          end
        end
        break
      end
    end
  end
  for i,v in base.pairs(coal.country) do
    if v.name == country_name then
      return v, coal
    end
  end
end

-------------------------------------------------------------------------------
function set_group_color(group, color)
  if group.mapObjects.units then
    local units = group.mapObjects.units
    for i=1,#units do
      units[i].currColor = color
    end
  end
  if group.mapObjects.route then
    local route = group.mapObjects.route
    route.line.currColor = color
    for i=1,#route.points do
      route.points[i].currColor = color
      if route.points[i].targets then
        for j=1,#route.points[i].targets do
          route.points[i].targets.points[j].currColor = color
          route.points[i].targets.numbers[j].currColor = color
        end
      end
    end
  end
end

-------------------------------------------------------------------------------
function set_waypoint_color(wpt, color)
  local group = wpt.boss
  group.mapObjects.route.points[wpt.index].currColor = color
  group.mapObjects.route.numbers[wpt.index].currColor = color
end

-------------------------------------------------------------------------------
function set_waypoints_color(group, color)
  local points = group.mapObjects.route.points
  for i=1,#points do
    points[i].currColor = color
  end
  local numbers = group.mapObjects.route.numbers
  for i=1,#numbers do
    numbers[i].currColor = color
  end
end

-------------------------------------------------------------------------------
function set_route_line_color(group, color)
  group.mapObjects.route.line.currColor = color
end

-------------------------------------------------------------------------------
function set_unit_color(unit, color)
  local group = unit.boss
  group.mapObjects.units[unit.index].currColor = color
end

-------------------------------------------------------------------------------
function set_target_color(target, color)
  local group = target.boss.boss
  group.mapObjects.route.targets[target.boss.index][target.index].currColor = color
  group.mapObjects.route.targetLines[target.boss.index][target.index].currColor = color
  group.mapObjects.route.targetNumbers[target.boss.index][target.index].currColor = color
end

-------------------------------------------------------------------------------
function set_targets_color(wpt,  color)
  local targets = wpt.boss.mapObjects.route.targets[wpt.index]
  if not targets then
      return
  end
  for i,v in base.pairs(targets) do
    v.currColor = color
  end
  local targetLines = wpt.boss.mapObjects.route.targetLines[wpt.index]
  for i,v in base.pairs(targetLines) do
    v.currColor = color
  end
  local targetNumbers = wpt.boss.mapObjects.route.targetNumbers[wpt.index]
  for i,v in base.pairs(targetNumbers) do
    v.currColor = color
  end
end

-------------------------------------------------------------------------------
function addList(fName)
  mapView:addList(fName)
end

-------------------------------------------------------------------------------
function addZip(fName)
  mapView:addZip(fName)
end

-------------------------------------------------------------------------------
function addBuildings(fName)
  --base.U.Stack();
  mapView:addSQRZipBin(fName, 'S0043100000')
end

-------------------------------------------------------------------------------
function getScales()
  return scales
end

-------------------------------------------------------------------------------
function getScale()
  return mapView:getScale()
end

-------------------------------------------------------------------------------
function setScale(s)
  mapView:setScale(s)
end

-------------------------------------------------------------------------------
function getLayers()
  return mapView:getLayers()
end

-------------------------------------------------------------------------------
function getObjects()
  return mapView:getObjects()
end

-------------------------------------------------------------------------------
function showLayer(t, b)
  mapView:showLayer(t, b)
  mapView:updateUserList(false)
end

-------------------------------------------------------------------------------
function getCamera()
   return mapView:getCamera()
end

-------------------------------------------------------------------------------
function setCamera(x, y)
  mapView:setCamera(x, y)
  mapView:updateUserList(false)
end

-------------------------------------------------------------------------------
function setPanState()
  mapView:setState(panState)
  base.statusbar.updateState()
end

-------------------------------------------------------------------------------
function setTapeState()
  mapView:setState(tapeState)
  base.statusbar.updateState()
end

-------------------------------------------------------------------------------
function setSelectState()
  mapView:setState(selState)
  base.statusbar.updateState()
end

-------------------------------------------------------------------------------
function showBkgImage(show)
  mapView:showBkgImage(show)
end

-------------------------------------------------------------------------------
function show(b)
    if b then
        if not created then
            doCreate(base.unpack(startParameters))
        end;
        if not loaded then
            load()
            loaded = true;
        end;
        if Mission.mission and not Mission.mission.mapElementsCreated then 
            Mission.createMapElements();
        end;
    end;
    if window then
        window:setVisible(b)
    end;
end

-------------------------------------------------------------------------------
-- обновление  панелей
function openPanels(group, vd, rvd, tvd, pvd, svd)
    local panel;
    if group.type == 'static' then
        selectedUnit = nil
        panel = base.panel_static

        vd = panel.vdata
        vd.group = group
        vd.name = group.name
        vd.category = group.units[1].category
        vd.type = group.units[1].type
        vd.lat = group.lat
        vd.long = group.long
        vd.heading = group.heading*base.math.pi/180
    else
        if (group.type == 'plane') or (group.type == 'helicopter') then
            panel = base.panel_aircraft
            panel.setView(group.type);
        elseif group.type == 'ship' then
            panel = base.panel_ship
        elseif group.type == 'vehicle' then
            panel = base.panel_vehicle
        end

		if (base.isPlannerMission() ~= true) then
			panel.setSafeMode(false)
		end
			
        base.panel_route.setSafeMode(false)

        vd = panel.vdata
        vd.group = group
        vd.name = group.name
        vd.country = group.boss.name
        vd.task = group.task
        vd.type = group.units[1].type
        vd.unit.number = #group.units
        if selectedUnit then
            vd.unit.cur = selectedUnit.index
        else
            vd.unit.cur = 1
        end

        for i=1,#group.units do
            local u = group.units[i]
            if group.type == 'plane' or group.type == 'helicopter' then
                vd.pilots[i] = u.name
                vd.type = u.type
            else
                vd.types[i] = u.type
            end
            vd.skills[i] = u.skill
        end

        rvd = base.panel_route.vdata
        rvd.group = group

        tvd = base.panel_targeting.vdata
        tvd.group = group

        svd = base.panel_summary.vdata
        svd.group = group
        svd.start_time = base.module_mission.mission.start_time + group.start_time

        base.panel_summary.update()

        if group.type == 'plane' or group.type == 'helicopter' then
            pvd = base.panel_payload.vdata
            pvd.group = group
            pvd.unit = group.units[vd.unit.cur]
        end
    end

    if not panel.window:isVisible() then
        base.toolbar.untoggle_all_except(base.toolbar.b_unit_list)
    end

    if (group.type == 'ship') or (group.type == 'vehicle') then
        base.panel_route.enableAlt(false)
    end

    if vd.unit then
        vd.unit.number = #group.units
    end
    panel.update();
    return vd, rvd, tvd, pvd, svd, panel
end;

-------------------------------------------------------------------------------
-- реагирование на действия мышой
function performActions(obj, selectedUnit, group, vd, rvd, tvd)
    local firstWptClsId
    local switched = false
    if (obj.waypoint) and
        group.route and
        group.route.points and
        group.route.points[1] and
        not selectedUnit 
    then
        -- waypoint or ...
        local wptA
        if selectedUnit then
            wptA = group.route.points[1]
        else
            wptA = obj.userObject
        end
        -- Точка маршрута
        if vd.tabs then
            vd.tabs:selectTab(1)
            base.panel_route.show(true)
            base.panel_targeting.show(false)
            group.mapObjects.route.points[wptA.index].currColor = group.boss.boss.selectWaypointColor
            group.mapObjects.route.numbers[wptA.index].currColor = group.boss.boss.selectWaypointColor
            group.mapObjects.route.line.currColor = group.boss.boss.selectGroupColor
            
            base.panel_route.vdata.wpt = wptA
            base.panel_route.b_add:setState(false)
            base.panel_route.b_edit:setState(true)
            base.panel_route.update()
            --base.print('route update')
            switched = true
        end;
    elseif obj.classKey == 'P0091000044' then
        -- Цель
        --base.panel_targeting.selectTargetByIndex(0); -- сброс выделения целей для предыдущей точки маршрута
        base.panel_targeting.selectTarget(nil); -- сброс выделения целей для предыдущей точки маршрута
        vd.tabs:selectTab(2)
        local target = obj.userObject
        local wpt = target.boss
        tvd.target = target;
        rvd.wpt = wpt;
        base.panel_targeting.show(true)
        base.panel_route.show(false)
        
        group.mapObjects.route.points[wpt.index].currColor = group.boss.boss.selectWaypointColor
        group.mapObjects.route.numbers[wpt.index].currColor = group.boss.boss.selectWaypointColor
        group.mapObjects.route.line.currColor = group.boss.boss.selectGroupColor
        --base.panel_route.update()
        base.panel_targeting.selectTarget(target);
        
        base.panel_targeting.b_add:setState(false)
        base.panel_targeting.b_edit:setState(true)
        base.panel_targeting.update(false)
        switched = true
    elseif obj.classKey == 'P0091000206' then -- INU Fix points
        vd.tabs:selectTab(5)
        base.panel_route.show(false)
        base.panel_targeting.show(false)
        base.panel_payload.show(false)
        base.panel_summary.show(false)
        base.panel_fix_points.show(true)
        obj.currColor = group.boss.boss.selectWaypointColor;
        for k,v in base.pairs(group.mapObjects.INUFixPoints) do 
            if v == obj then
                group.mapObjects.INUFixPoints_numbers[k].currColor = group.boss.boss.selectWaypointColor;
                base.panel_fix_points.vdata.selectedPoint = group.INUFixPoints[k];
                base.panel_fix_points.setEditMode(true);
                base.panel_fix_points.update();
                break;
            end;
        end;
        switched = true
        if base.panel_route.vdata.wpt ~= nil then
            local wpt = base.panel_route.vdata.wpt;
            --base.print('wpt',base.panel_route.vdata.wpt);
            group.mapObjects.route.points[wpt.index].currColor = group.boss.boss.selectWaypointColor;
            group.mapObjects.route.numbers[wpt.index].currColor = group.boss.boss.selectWaypointColor;
        end;
    end
    
    -- Чтобы не путать с линиями, подписями и зонами.
    -- Юнит
    --base.print('selectedUnit', selectedUnit);base.U.trav(selectedUnit)
    if selectedUnit and group.mapObjects.units[obj.userObject.index] then
        group.mapObjects.units[obj.userObject.index].currColor = group.boss.boss.selectUnitColor
        group.mapObjects.route.numbers[1].currColor = group.boss.boss.selectWaypointColor;
        base.panel_route.vdata.wpt = group.route.points[1];
        base.panel_route.update()
    end
    if group.type ~= 'static' then
        if not switched and vd.tabs then
            vd.tabs:selectTab(1)
        end
        tvd.group = group
        -- Соответствующий номер юнита нужно установить в панели группы.
        if selectedUnit then
            vd.unit.cur = obj.userObject.index
        end
        base.panel_route.b_add:setState(false)
        base.panel_route.b_edit:setState(true)
        base.panel_route.vdata.mode = 'EDIT';
        mapView:setState(panState);
        if (obj.classKey ~= 'P0091000044') and (obj.classKey ~= 'P0091000206') then
            base.panel_route.show(true)
        end
        
        --base.print('route update')
    end
    base.panel_route.update() -- обновляем маршрут всегда
end;



-------------------------------------------------------------------------------
-- реакция на выбор группы
function respondToSelectedUnit(obj, group, _selectedUnit)
   -- base.print('respondToSelectedUnit: obj, group, selectedUnit',obj, group, selectedUnit);
   -- base.print('selectedUnit=', _selectedUnit);
    --base.debug.sethook ( base.hookFcn, 'c' );
    selectedGroup = group
    selectedUnit = _selectedUnit;
    selectedObject = obj.id
    base.panel_fix_points.vdata.selectedPoint = nil;
    --base.panel_targeting.vdata.target = nil;
    
    -- запоминаем смещение юнитов группы отн-но головного для последующего таскания
    if selectedGroup and selectedGroup.route and selectedGroup.route.points then
        local wayPoint = selectedGroup.route.points[1];
        selectedGroup.units[1].dLat = 0;
        selectedGroup.units[1].dLong = 0;
        for i = 2, #selectedGroup.units do
            local unit = selectedGroup.units[i];
            unit.dLat = unit.lat - wayPoint.lat;
            unit.dLong = unit.long - wayPoint.long;
        end;
    end;
    
    -- В зависимости от типа группы, нужно открыть соответствующие диалоги
    -- и проинициализировать в них таблицы и контролы.
    local vd, rvd, tvd, pvd, svd, panel = {},{},{},{},{},{}
    vd, rvd, tvd, pvd, svd, panel = openPanels(group, vd, rvd, tvd, pvd, svd);
    -- Затем нужно покрасить объекты группы в цвет выбора.
    set_selected_group_color(group)
    -- Затем нужно разобраться, что именно выбрали - юнит, точку маршрута или цель -
    -- и соответственно изменить цвет выбранного объекта.

    performActions(obj, selectedUnit, group, vd, rvd, tvd)
    --panel.update()

    --base.print('group panel update')
    panel.show(true)
    -- Кроме того, если выбрана точка маршрута, то координаты нужно
    -- сохранить для возможного отката после перемещения, если тип
    -- поверхности не подойдет.
    -- Наконец, нужно не забыть сообщить карте об изменениях цветов.
    base.module_mission.update_group_map_objects(group)
    --base.debug.sethook( nil);
end;

-------------------------------------------------------------------------------
-- обработчик чекбокса показа/скрытия юнита
function OnHiddenCheckboxChange(self)
    if selectedGroup then     -- нет выбранной группы
        selectedGroup.hidden = self:getState();
        -- обновить инфо группы
        base.panel_units_list.updateRow(selectedGroup);
        if selectedGroup.hidden then -- группа спрятана, удаляем  группу с карты
            Mission.remove_group_map_objects(selectedGroup)
        else   -- группа видна, добавляем  группу на карты
            Mission.create_group_map_objects(selectedGroup)
            -- выделяем группу на карте
            base.panel_units_list.selectGroup(selectedGroup);
        end;
        -- обновляем карту
        mapView:updateUserList(true)
    else
        self:setState(false)
    end;
end;

-------------------------------------------------------------------------------
function OnTrigZoneHiddenCheckboxChange(self)
    if base.panel_zone.currentZone 
        and base.module_mission.mission.triggers 
        and base.module_mission.mission.triggers.zones 
    then
        local zone = base.panel_zone.currentZone
        zone.hidden = self:getState();
        base.panel_trig_zones_list.updateZone(zone);
        if zone.hidden then
            mapView:removeUserObjects(zone.mapObjects)
        else
            Mission.create_trigger_zone_map_object(zone)
        end;
        mapView:updateUserList(true)
    else
        self:setState(false)
    end;
end;

-------------------------------------------------------------------------------
-- recreate selected group objects
function updateSelectedGroup(group)
    base.module_mission.remove_group_map_objects(group)
    base.module_mission.create_group_map_objects(group)
    set_selected_group_color(group)
    set_waypoint_color(group.route.points[base.panel_route.vdata.wpt.index], 
        group.boss.boss.selectWaypointColor)
    base.module_mission.update_group_map_objects(group)
    mapView:updateUserList(true)
end

-------------------------------------------------------------------------------
function createAircraft(aircraftType, x, y)
    unselectAll()
    base.panel_aircraft.setSafeMode(false)
    base.panel_route.setSafeMode(false)
    base.toolbar.form.b_helicopter:setState(false)
    base.toolbar.form.b_airplane:setState(false)
    local lat, long = mapView:getMapPoint(x, y)
    if not isInMap(lat, long) then
      showWarningWindow(cdata.placeInside)
      return
    end

    local vd = base.panel_aircraft.vdata

    local rvd = base.panel_route.vdata
    local tvd = base.panel_targeting.vdata
    local svd = base.panel_summary.vdata
    local pvd = base.panel_payload.vdata
    -- При добавлении точек группа должна быть известна
    local group = base.module_mission.create_group(vd.country, aircraftType, vd.name, base.panel_briefing.vdata.start_time, lat, long)
    
    group.task = vd.task
	group.defaultTask = vd.defaultTask
    selectedGroup = group
    vd.group = group
    vd.name = group.name
    vd.unit.cur = 1
    vd.unit.number = 1
    rvd.group = group
    tvd.group = group
    tvd.target = nil
    svd.group = group
    pvd.group = group

    local skill = vd.skills[1];
    if (skill == crutches.getPlayerSkill())
        and base.module_mission.playerUnit then -- если юнит играбельный и юнит с игроком уже есть, то
        -- надо проследить, чтобы не возникло двух юнитов со скилом игрока
        skill = base.panel_aircraft.aiSkills[1];
    end; 
    
    local unit = base.module_mission.insert_unit(group, vd.type, skill, 1, vd.pilots[1])
    
    local name = unit.name
    vd.pilots = { name }
    vd.skills = { skill}
    pvd.unit = unit
    
    local height = U.getAltitude(lat, long)
    local alt = base.math.max(height, initialAltitude[aircraftType])
    rvd.wpt = base.module_mission.insert_waypoint(group, 1, 
          base.panel_route.actions.turningPoint, lat, long, alt, initialVelocity[aircraftType])
    rvd.wpt.eta = base.panel_briefing.vdata.start_time
    set_group_color(group, group.boss.boss.selectGroupColor)
    set_unit_color(unit, group.boss.boss.selectUnitColor)
    set_waypoint_color(rvd.wpt, group.boss.boss.selectWaypointColor)
    set_route_line_color(group, group.boss.boss.selectGroupColor)
    
    base.module_mission.update_group_map_objects(group)
    mapView:updateUserList(true)

    base.panel_aircraft.vdata.group = group
	base.panel_aircraft.updateAutoTask()

    mapView:setState(addingWaypointState)
    base.statusbar.updateState()
    base.panel_route.b_add:setState(true)
    base.panel_route.b_edit:setState(false)
    base.panel_aircraft.update()
    base.panel_targeting.update(false)
    base.panel_route.update()
end;
